/*:
 * @target MZ MV
 * @plugindesc A localization system that replaces game text based on CSV data. The best localization plugin <3
 * @author Sang Hendrix
 * @url https://sanghendrix.itch.io/
 * 
 * @help
 * Verion 2.0.0
 * ----------------------------------------------------------------------------
 * This RPG Maker MV/MZ plugin allows you to translate everything you
 * see from your game using a auto generated CSV file, and you don't need to
 * touch your in-editor text. Totally un-destructive method and automatic.
 * All you need to do is to translate, the plugin handles the rest.
 * ----------------------------------------------------------------------------
 * HOW TO USE
 * ----------------------------------------------------------------------------
 * Both methods extract all contents in game to game_messages.csv file,
 * choose method 1 if you want to use Smart Sheet Editor.
 *  
 * Method 1:
 * Play your game > Press Sheet Editor button (see in parameter)
 * > Click Extract CSV button > Translate your game
 * 
 * Method 2:
 * Turn on Extract Text parameter > Start the game > Hit Yes to generate
 * all context from your game to a translation file (game_messages.csv)
 * 
 * Tutorials: https://www.youtube.com/watch?v=QvHeVdmUjRQ
 * ----------------------------------------------------------------------------
 * FAQ
 * ----------------------------------------------------------------------------
 * - What if I want to translate something manually?
 * Then open game_messages.csv and type it in <3.
 * 
 * - I see lots of columns in the csv file, what are they?
 * Change column: To tell you which line is Newly added to your game.
 * Original: Whatever is written in here will be translated.
 * Languages column (en, vi, jp, es...): Put you translation here.
 * 
 * - How to check the current language using conditional branch?
 * Simply put this: ConfigManager.language === 'en' (en is your symbol)
 * 
 * - I'm gonna localize pictures so I'm gonna use ConfigManager.language...
 * STOP! This plugin is created so you don't do any manual work.
 * Watch this video: https://www.youtube.com/watch?v=V0e5w1c0tgE
 * ----------------------------------------------------------------------------
 * SCRIPT CALL
 * ----------------------------------------------------------------------------
 * To change to a specific language: changeToLanguage('languageSymbol')
 * Example: changeToLanguage('en') or changeToLanguage('jp')
 * To change to the next avaiable language: changeToLanguage('next')
 * ----------------------------------------------------------------------------
 * For support, please contact me via Discord:
 * https://discord.com/invite/YKPscqHV8b
 * ----------------------------------------------------------------------------
 * 
 * @param Sheet Editor Button
 * @type select
 * @option none
 * @option pagedown
 * @option pageup
 * @option control
 * @option shift
 * @desc The button to press to open the Sheet Editor
 * @default control
 * 
 * @param decor1dsf
 * @text ==========================
 * @type text
 * @default ===============================
 * 
 * @param Exact Text
 * @text Extract Text
 * @type boolean
 * @desc Extracts text from games (choose what to extract below)
 * @default false
 * 
 * @param decor1
 * @text ==========================
 * @type text
 * @default ===============================
 * 
 * @param Adding Maps name
 * @type boolean
 * @desc Extract map names when extracting messages
 * @default true
 * 
 * @param Extract Map Display Names
 * @type boolean
 * @desc Extracts map display names
 * @default false
 * 
 * @param Extract Names (MZ)
 * @text Extract Speaker Name (MZ)
 * @type boolean
 * @desc Extracts the name box from Show Message commands to a separate CSV column. Only for MZ
 * @default false
 * 
 * @param Extract Database Entries
 * @type boolean
 * @desc Extracts database entries
 * @default false
 * 
 * @param Extract Variable Text
 * @type boolean
 * @desc Extracts the script text of all variables
 * @default false
 * 
 * @param Extract Show Picture
 * @type boolean
 * @desc Extracts the default filename of all Show Pictures commands
 * @default false
 * 
 * @param Extract Play Movies
 * @type boolean
 * @desc Extracts the filename of all Play Movie commands
 * @default false
 * 
 * @param Extract Script Call
 * @type boolean
 * @desc Extracts text from script calls
 * @default false
 * 
 * @param Extract Plugin Command Text
 * @type boolean
 * @desc Extracts text from plugin commands (string, text, note types)
 * @default false
 * 
 * @param Extract Comments
 * @type boolean
 * @desc Extracts comments (the green text)
 * @default false
 * 
 * @param Exclude Name Text
 * @text Exclude Text Before ":"
 * @type boolean
 * @desc Excludes all text from extraction before the character ":" ("David: hehe" --csv-> "hehe"). Don't True this if you don't know what this is
 * @default false
 * 
 * @param decor2
 * @text ==========================
 * @type text
 * @default ===============================
 * 
 * @param Partial Matching
 * @type boolean
 * @desc Enables partial matching for translations. Best to be false
 * @default false
 * 
 * @param Use Translation Cache
 * @type boolean
 * @desc Caches translated results for frequently used text. Heavily improve performance
 * @default false
 * 
 * @param Auto Refresh Pictures
 * @type boolean
 * @desc Automatically refreshes pictures when changing language. Required if you localize your pictures
 * @default false
 * 
 * @param decor3
 * @text ==========================
 * @type text
 * @default ===============================
 *
 * @param Languages
 * @type struct<Language>[]
 * @desc List of available languages
 *
 * @param Default Language
 * @type string
 * @desc The symbol of the default language to use for translations
 * @default en
 * 
 * @param Add Language Command to Title
 * @text Add Language to Title
 * @type boolean
 * @desc Adds a language selection command to the title screen
 * @default false
 * 
 * @param decor4
 * @text ==========================
 * @type text
 * @default ===============================
 * 
 * @param Excluded Symbols
 * @type struct<PushSymbol>[]
 * @desc Symbols to push to Excluded column. Example: \n<*>, \pop[*]. Only for advanced usage
 * @default []
 * 
 */

/*~struct~PushSymbol:
 * @param Symbol
 * @type string
 * @desc The symbol used to identify text to push to Excluded column (e.g. \n<*>)
 * @default \n<*>
 */

/*~struct~Language:
 * @param Name
 * @type string
 * @desc The display name of the language
 *
 * @param Symbol
 * @type string
 * @desc The symbol used to identify this language (e.g. en, vi)
 *
 * @param Font
 * @type string
 * @desc The name of the font file in the fonts folder (e.g. myfont.ttf)
 *
 * @param FontSize
 * @type number
 * @min 1
 * @desc The font size to use for this language
 * @default
 */

var Imported = Imported || {};
Imported.Hendrix_Localization = true;

(function () {
    let fs, path;
    if (typeof require === 'function') {
        fs = require('fs');
        path = require('path');
    } else {
        fs = null;
        path = null;
    }
    const parameters = PluginManager.parameters('Hendrix_Localization');
    const extractComments = parameters['Extract Comments'] === 'true';
    const exactText = parameters['Exact Text'] === 'true' && Utils.isOptionValid('test');
    const extractDatabaseEntries = parameters['Extract Database Entries'] === 'true';
    const addingMapsName = parameters['Adding Maps name'] === 'true';
    const extractNamesMZ = parameters['Extract Names (MZ)'] === 'true';
    const excludeNameText = parameters['Exclude Name Text'] === 'true';
    const addGibberishLines = parameters['Add Gibberish Lines'] === 'true';
    const extractVariableText = parameters['Extract Variable Text'] === 'true';
    const extractPluginCommandText = parameters['Extract Plugin Command Text'] === 'true';
    const defaultLanguage = parameters['Default Language'];
    const partialMatching = parameters['Partial Matching'] === 'true';
    const useTranslationCache = parameters['Use Translation Cache'] === 'true';
    const languagesParam = JSON.parse(parameters['Languages'] || '[]');
    const extractShowPicture = parameters['Extract Show Picture'] === 'true';
    const autoRefreshPictures = parameters['Auto Refresh Pictures'] === 'true';
    const pushSymbols = JSON.parse(parameters['Excluded Symbols'] || '[]').map(symbol => JSON.parse(symbol).Symbol);
    const addLanguageCommandToTitle = parameters['Add Language Command to Title'] === 'true';
    const extractPlayMovies = parameters['Extract Play Movies'] === 'true';
    const extractScriptCall = parameters['Extract Script Call'] === 'true';
    const extractMapDisplayNames = parameters['Extract Map Display Names'] === 'true';
    const sheetEditorButton = parameters['Sheet Editor Button'] || 'pagedown';

    const languages = languagesParam.map(lang => JSON.parse(lang));
    const languageSymbols = languages.map(lang => lang.Symbol);

    let currentLanguage = defaultLanguage;
    let availableLanguages = [];
    let messages = [];
    let translations = {};
    let translationCache = {};
    let originalTitle1 = '';
    let originalTitle2 = '';
    const wordBoundaryRegex = /\b/g;
    let uniqueMessages = new Set();
    const originalTexts = [];
    const wordTranslationRegexes = new Map();
    const messageData = new Map();

    function detectCSVSeparator() {
        const numberWithDecimal = 1.1;
        const formatted = numberWithDecimal.toLocaleString();
        const usesSemicolon = formatted.includes(',');
        return usesSemicolon ? ';' : ',';
    }

    function detectCSVSeparatorFromContent(content) {
        const lines = content.split('\n').slice(0, 5);
        if (lines.length > 0) {
            const header = lines[0];
            const commaCount = (header.match(/,/g) || []).length;
            const semicolonCount = (header.match(/;/g) || []).length;
            const detectedSeparator = semicolonCount > commaCount ? ';' : ',';

            return detectedSeparator;
        }
        return ',';
    }

    const CSV_SEPARATOR = detectCSVSeparator();

    function escapeRegExp(string) {
        return string.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
    }

    function extractYepName(text) {
        let yepName = '';
        let remainingText = text;

        for (const symbol of pushSymbols) {
            const escapedSymbol = escapeRegExp(symbol);
            const regex = new RegExp(escapedSymbol.replace('\\*', '(.+?)'), 'g');
            let match;
            while ((match = regex.exec(remainingText)) !== null) {
                yepName += match[0];
                remainingText = remainingText.replace(match[0], '');
            }
        }

        return { yepName, remainingText };
    }

    function generateGibberish(length) {
        const characters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789';
        let result = '';
        for (let i = 0; i < length; i++) {
            result += characters.charAt(Math.floor(Math.random() * characters.length));
        }
        return result;
    }

    const _Scene_Boot_start = Scene_Boot.prototype.start;
    Scene_Boot.prototype.start = function () {
        _Scene_Boot_start.call(this);
        if (Imported['YEP_MessageCore']) {
            this.updateMessageCoreEscapeCharacters();
        }
        if (exactText) {
            setTimeout(extractAllMessages, 1000);
        }
        loadTranslations(currentLanguage);
        applyLanguageSettings();
    };

    Scene_Boot.prototype.updateMessageCoreEscapeCharacters = function () {
        Game_Message.prototype.addText = function (text) {
            if ($gameSystem.wordWrap()) text = '<WordWrap>' + Hendrix_Localization(text);
            this.add(text);
        };

        Window_Base.prototype.setWordWrap = function (text) {
            this._wordWrap = false;
            if (text.match(/<(?:WordWrap)>/i)) {
                this._wordWrap = true;
                text = text.replace(/<(?:WordWrap)>/gi, '');
            }
            if (this._wordWrap) {
                var replace = Yanfly.Param.MSGWrapSpace ? ' ' : '';
                text = Hendrix_Localization(text).replace(/[\n\r]+/g, replace);
            }
            if (this._wordWrap) {
                text = Hendrix_Localization(text).replace(/<(?:BR|line break)>/gi, '\n');
            } else {
                text = Hendrix_Localization(text).replace(/<(?:BR|line break)>/gi, '');
            }
            return text;
        };

        Window_Base.prototype.escapeIconItem = function (n, database) {
            return '\x1bI[' + database[n].iconIndex + ']' + Hendrix_Localization(database[n].name);
        };

        var _Window_Base_convertExtraEscapeCharacters = Window_Base.prototype.convertExtraEscapeCharacters;
        Window_Base.prototype.convertExtraEscapeCharacters = function (text) {
            text = _Window_Base_convertExtraEscapeCharacters.call(this, text);
            // \AC[n]
            text = text.replace(/\x1bAC\[(\d+)\]/gi, function () {
                return Hendrix_Localization(this.actorClassName(parseInt(arguments[1])));
            }.bind(this));
            // \AN[n]
            text = text.replace(/\x1bAN\[(\d+)\]/gi, function () {
                return Hendrix_Localization(this.actorNickname(parseInt(arguments[1])));
            }.bind(this));
            // \PC[n]
            text = text.replace(/\x1bPC\[(\d+)\]/gi, function () {
                return Hendrix_Localization(this.partyClassName(parseInt(arguments[1])));
            }.bind(this));
            // \PN[n]
            text = text.replace(/\x1bPN\[(\d+)\]/gi, function () {
                return Hendrix_Localization(this.partyNickname(parseInt(arguments[1])));
            }.bind(this));
            // \NC[n]
            text = text.replace(/\x1bNC\[(\d+)\]/gi, function () {
                return Hendrix_Localization($dataClasses[parseInt(arguments[1])].name);
            }.bind(this));
            // \NI[n]
            text = text.replace(/\x1bNI\[(\d+)\]/gi, function () {
                return Hendrix_Localization($dataItems[parseInt(arguments[1])].name);
            }.bind(this));
            // \NW[n]
            text = text.replace(/\x1bNW\[(\d+)\]/gi, function () {
                return Hendrix_Localization($dataWeapons[parseInt(arguments[1])].name);
            }.bind(this));
            // \NA[n]
            text = text.replace(/\x1bNA\[(\d+)\]/gi, function () {
                return Hendrix_Localization($dataArmors[parseInt(arguments[1])].name);
            }.bind(this));
            // \NE[n]
            text = text.replace(/\x1bNE\[(\d+)\]/gi, function () {
                return Hendrix_Localization($dataEnemies[parseInt(arguments[1])].name);
            }.bind(this));
            // \NS[n]
            text = text.replace(/\x1bNS\[(\d+)\]/gi, function () {
                return Hendrix_Localization($dataSkills[parseInt(arguments[1])].name);
            }.bind(this));
            // \NT[n]
            text = text.replace(/\x1bNT\[(\d+)\]/gi, function () {
                return Hendrix_Localization($dataStates[parseInt(arguments[1])].name);
            }.bind(this));
            // \II[n]
            text = text.replace(/\x1bII\[(\d+)\]/gi, function () {
                return this.escapeIconItem(arguments[1], Hendrix_Localization($dataItems));
            }.bind(this));
            // \IW[n]
            text = text.replace(/\x1bIW\[(\d+)\]/gi, function () {
                return this.escapeIconItem(arguments[1], Hendrix_Localization($dataWeapons));
            }.bind(this));
            // \IA[n]
            text = text.replace(/\x1bIA\[(\d+)\]/gi, function () {
                return this.escapeIconItem(arguments[1], Hendrix_Localization($dataArmors));
            }.bind(this));
            // \IS[n]
            text = text.replace(/\x1bIS\[(\d+)\]/gi, function () {
                return this.escapeIconItem(arguments[1], Hendrix_Localization($dataSkills));
            }.bind(this));
            // \IT[n]
            text = text.replace(/\x1bIT\[(\d+)\]/gi, function () {
                return this.escapeIconItem(arguments[1], Hendrix_Localization($dataStates));
            }.bind(this));

            return text;
        };
    };

    function applyLanguageSettings() {
        const currentLangSettings = languages.find(lang => lang.Symbol === currentLanguage);
        if (currentLangSettings) {
            if (currentLangSettings.Font && currentLangSettings.Font.trim() !== '') {
                loadCustomFont(currentLangSettings.Font);
            }

            if (currentLangSettings.FontSize && Number(currentLangSettings.FontSize) > 0) {
                const fontSize = Number(currentLangSettings.FontSize);
                applyFontSize(fontSize);
            }

            if (Imported['YEP_MessageCore']) {
                overrideYanflyMessageFontSettings(currentLangSettings);
            }

            if (Imported['Galv_MessageStyles']) {
                alterGalvFontSize(currentLangSettings);
            }
        }
    }

    function overrideYanflyMessageFontSettings(langSettings) {
        if (langSettings.Font && langSettings.Font.trim() !== '') {
            Game_System.prototype.getMessageFontName = function () {
                return langSettings.Font.split('.')[0];
            };
        }
        if (langSettings.FontSize && Number(langSettings.FontSize) > 0) {
            Game_System.prototype.getMessageFontSize = function () {
                return Number(langSettings.FontSize);
            };
        }
    }

    function alterGalvFontSize(langSettings) {
        if (langSettings.FontSize && Number(langSettings.FontSize) > 0) {
            Galv.Mstyle.fontSize = Number(langSettings.FontSize);

            Window_Message.prototype.standardFontSize = function () {
                return Galv.Mstyle.fontSize;
            };

            Window_ChoiceList.prototype.standardFontSize = function () {
                return Galv.Mstyle.fontSize;
            };

            if (SceneManager._scene && SceneManager._scene.refreshAllWindows) {
                SceneManager._scene.refreshAllWindows();
            }
        }
    }

    function loadCustomFont(fontFileName) {
        if (!fontFileName || fontFileName.trim() === '') {
            return;
        }
        const fontName = fontFileName.split('.')[0];
        const fileExtension = fontFileName.split('.').pop().toLowerCase();
        let format;

        if (fileExtension === 'ttf') {
            format = 'truetype';
        } else if (fileExtension === 'otf') {
            format = 'opentype';
        } else {
            return;
        }

        const fontPath = `url('fonts/${fontFileName}')`;
        const font = new FontFace(fontName, fontPath);

        font.load().then((loadedFont) => {
            document.fonts.add(loadedFont);
            applyFontFace(fontName);
            if (Imported['YEP_MessageCore']) {
                Game_System.prototype.getMessageFontName = function () {
                    return fontName;
                };
            }
            if (SceneManager._scene && SceneManager._scene.refreshAllWindows) {
                SceneManager._scene.refreshAllWindows();
            }
        }).catch(() => { });

        const style = document.createElement('style');
        const css = `@font-face {
            font-family: "${fontName}";
            src: ${fontPath} format("${format}");
        }`;
        style.appendChild(document.createTextNode(css));
        document.head.appendChild(style);
    }

    function applyFontFace(fontFace) {
        Window_Base.prototype.standardFontFace = function () {
            return fontFace;
        };
        if (SceneManager._scene && SceneManager._scene.refreshAllWindows) {
            SceneManager._scene.refreshAllWindows();
        }
    }

    function applyFontSize(fontSize) {
        if (!fontSize || Number(fontSize) <= 0) {
            return;
        }

        Window_Base.prototype.standardFontSize = function () {
            return fontSize;
        };
        if (SceneManager._scene && SceneManager._scene.refreshAllWindows) {
            SceneManager._scene.refreshAllWindows();
        }
    }

    const _Window_Base_resetFontSettings = Window_Base.prototype.resetFontSettings;
    Window_Base.prototype.resetFontSettings = function () {
        _Window_Base_resetFontSettings.call(this);
        if (typeof this.standardFontFace === 'function') {
            this.contents.fontFace = this.standardFontFace();
        }
        if (typeof this.standardFontSize === 'function') {
            this.contents.fontSize = this.standardFontSize();
        }
    };

    function extractAllMessages() {
        if (!fs || !path) {
            return;
        }
        messages = [];
        if (confirm("Do you want to generate/update the csv file?")) {
            let filePath = 'game_messages.csv';
            if (!fs.existsSync(filePath)) {
                filePath = path.join('www', 'game_messages.csv');
            }

            if (fs.existsSync(filePath)) {
                if (confirm("Do you want to create a backup of your current localization file before proceeding?\n\nSelect 'OK' to make a backup, or 'Cancel' to proceed without a backup.")) {
                    const timestamp = new Date().toISOString().replace(/[:.]/g, '-');
                    const backupPath = path.join(path.dirname(filePath), `game_messages_backup_${timestamp}.csv`);
                    fs.copyFileSync(filePath, backupPath);
                    alert(`Backup created: ${backupPath}`);
                } else {

                }
            }

            for (let mapId = 1; mapId < $dataMapInfos.length; mapId++) {
                if ($dataMapInfos[mapId]) {
                    const mapData = JSON.parse(fs.readFileSync(path.join('data', `Map${String(mapId).padStart(3, '0')}.json`), 'utf8'));
                    extractMessagesFromMap(mapData, mapId);
                }
            }

            if (extractMapDisplayNames) {
                extractMapDisplayNamesFromData();
            }

            extractCommonEventMessages();

            if (extractDatabaseEntries) {
                extractDatabaseMessages();
                extractTroopMessages();
            }

            if (extractVariableText) {
                extractVariableMessages();
            }

            if (extractShowPicture) {
                extractShowPictureMessages();
            }

            if (extractPlayMovies) {
                extractPlayMovieMessages();
            }

            if (extractScriptCall) {
                extractScriptCallMessages();
            }

            if (extractPluginCommandText) {
                extractPluginCommandMessages();
            }

            updateMessagesCSV();
            alert("CSV file has been generated/updated successfully.");
        } else {

        }
    }

    function extractMapDisplayNamesFromData() {
        messages.push('\n[MAP DISPLAY NAMES]\n');

        for (let mapId = 1; mapId < $dataMapInfos.length; mapId++) {
            if ($dataMapInfos[mapId]) {
                const mapData = JSON.parse(fs.readFileSync(path.join('data', `Map${String(mapId).padStart(3, '0')}.json`), 'utf8'));
                if (mapData && mapData.displayName) {
                    messages.push(mapData.displayName);
                }
            }
        }
    }

    function extractScriptCallMessages() {
        messages.push('\n[SCRIPT CALLS]\n');

        function extractQuotedText(text) {
            const matches = text.match(/"([^"]+)"/g);
            if (matches) {
                return matches.map(match => match.slice(1, -1))
                    .filter(str => /[a-zA-Z]/.test(str)); // Only keep strings with at least one letter
            }
            return [];
        }

        function extractFromEventCommands(eventCommands) {
            let currentScript = [];
            eventCommands.forEach(command => {
                if (command.code === 355 || command.code === 655) { // Script
                    let scriptText = command.parameters[0];
                    if (typeof scriptText === 'string') {
                        currentScript.push(scriptText);
                    }
                } else if (currentScript.length > 0) {
                    const fullScript = currentScript.join('\n');
                    const quotedTexts = extractQuotedText(fullScript);
                    quotedTexts.forEach(text => messages.push(text));
                    currentScript = [];
                }
            });
            if (currentScript.length > 0) {
                const fullScript = currentScript.join('\n');
                const quotedTexts = extractQuotedText(fullScript);
                quotedTexts.forEach(text => messages.push(text));
            }
        }

        for (let mapId = 1; mapId < $dataMapInfos.length; mapId++) {
            if ($dataMapInfos[mapId]) {
                const mapData = JSON.parse(fs.readFileSync(path.join('data', `Map${String(mapId).padStart(3, '0')}.json`), 'utf8'));
                if (mapData.events) {
                    mapData.events.forEach(event => {
                        if (event && event.pages) {
                            event.pages.forEach(page => {
                                if (page.list) {
                                    extractFromEventCommands(page.list);
                                }
                            });
                        }
                    });
                }
            }
        }

        $dataCommonEvents.forEach(commonEvent => {
            if (commonEvent && commonEvent.list) {
                extractFromEventCommands(commonEvent.list);
            }
        });
    }

    function extractCommonEventMessages() {
        messages.push('\n[Common Events]\n');
        $dataCommonEvents.forEach((commonEvent, index) => {
            if (commonEvent && commonEvent.list) {
                if (extractComments) {
                    let hasAddedCommentHeader = false;
                    commonEvent.list.forEach(command => {
                        if (command.code === 108 || command.code === 408) {
                            if (!hasAddedCommentHeader) {
                                messages.push('\n[Comments]\n');
                                hasAddedCommentHeader = true;
                            }
                            messages.push(command.parameters[0]);
                        }
                    });
                }

                let currentMessage = [];
                let currentSpeaker = '';

                commonEvent.list.forEach(command => {
                    if (command.code === 101) {
                        if (currentMessage.length > 0) {
                            const fullMessage = currentMessage.join('\n');
                            messages.push(fullMessage);
                            if (extractNamesMZ) {
                                messageData.set(fullMessage, currentSpeaker);
                            }
                            currentMessage = [];
                        }
                        currentSpeaker = command.parameters[4] || '';
                    } else if (command.code === 401) {
                        let text = command.parameters[0];
                        if (excludeNameText) {
                            const colonIndex = text.indexOf(':');
                            if (colonIndex !== -1) {
                                text = text.substring(colonIndex + 1).trim();
                            }
                        }
                        currentMessage.push(text);
                    } else if (command.code === 102) {
                        if (currentMessage.length > 0) {
                            const fullMessage = currentMessage.join('\n');
                            messages.push(fullMessage);
                            if (extractNamesMZ) {
                                messageData.set(fullMessage, currentSpeaker);
                            }
                            currentMessage = [];
                        }
                        command.parameters[0].forEach(choice => {
                            messages.push(choice);
                            if (extractNamesMZ) {
                                messageData.set(choice, '');
                            }
                        });
                    } else if (currentMessage.length > 0) {
                        const fullMessage = currentMessage.join('\n');
                        messages.push(fullMessage);
                        if (extractNamesMZ) {
                            messageData.set(fullMessage, currentSpeaker);
                        }
                        currentMessage = [];
                        currentSpeaker = '';
                    }
                });

                if (currentMessage.length > 0) {
                    const fullMessage = currentMessage.join('\n');
                    messages.push(fullMessage);
                    if (extractNamesMZ) {
                        messageData.set(fullMessage, currentSpeaker);
                    }
                }
            }
        });
    }

    function extractVariableMessages() {
        messages.push('\n[VARIABLES]\n');

        function extractFromEventCommands(eventCommands) {
            eventCommands.forEach(command => {
                if (command.code === 122 && command.parameters[3] === 4) {
                    const script = command.parameters[4];
                    if (typeof script === 'string' && /[a-zA-Z]/.test(script)) {
                        const cleanScript = script.replace(/^["']|["']$/g, '');
                        messages.push(cleanScript);
                    }
                }
            });
        }

        for (let mapId = 1; mapId < $dataMapInfos.length; mapId++) {
            if ($dataMapInfos[mapId]) {
                const mapData = JSON.parse(fs.readFileSync(path.join('data', `Map${String(mapId).padStart(3, '0')}.json`), 'utf8'));
                if (mapData.events) {
                    mapData.events.forEach(event => {
                        if (event && event.pages) {
                            event.pages.forEach(page => {
                                if (page.list) {
                                    extractFromEventCommands(page.list);
                                }
                            });
                        }
                    });
                }
            }
        }

        $dataCommonEvents.forEach(commonEvent => {
            if (commonEvent && commonEvent.list) {
                extractFromEventCommands(commonEvent.list);
            }
        });
    }

    function extractDatabaseMessages() {
        messages.push('\n[DATABASE ENTRIES]\n');

        messages.push('\n[GAME TITLE]\n');
        if ($dataSystem && $dataSystem.gameTitle) {
            messages.push($dataSystem.gameTitle);
        }

        messages.push('\n[Title Screen]\n');
        if ($dataSystem && $dataSystem.title1Name) {
            const title1Filename = $dataSystem.title1Name + '.png';
            messages.push(title1Filename);
        }
        if ($dataSystem && $dataSystem.title2Name) {
            const title2Filename = $dataSystem.title2Name + '.png';
            messages.push(title2Filename);
        }

        function removePrefix(text) {
            return text && typeof text === 'string' ? text.replace(/^\[.*?\]\s*/, '') : '';
        }

        messages.push('\n[Actors]\n');
        $dataActors.forEach((actor, index) => {
            if (actor) {
                if (actor.name) messages.push(actor.name);
                if (actor.nickname) messages.push(actor.nickname);
                if (actor.profile) messages.push(actor.profile);
            }
        });

        messages.push('\n[Classes]\n');
        $dataClasses.forEach((klass, index) => {
            if (klass && klass.name) {
                messages.push(klass.name);
            }
        });

        messages.push('\n[Skills]\n');
        $dataSkills.forEach((skill, index) => {
            if (skill && skill.name) {
                messages.push(removePrefix(skill.name));
                if (skill.description) messages.push(skill.description);
                if (skill.message1) messages.push(skill.message1);
                if (skill.message2) messages.push(skill.message2);
            }
        });

        messages.push('\n[Items]\n');
        $dataItems.forEach((item, index) => {
            if (item && item.name) {
                messages.push(item.name);
                if (item.description) messages.push(item.description);
            }
        });

        messages.push('\n[Weapons]\n');
        $dataWeapons.forEach((weapon, index) => {
            if (weapon && weapon.name) {
                messages.push(removePrefix(weapon.name));
                if (weapon.description) messages.push(weapon.description);
            }
        });

        messages.push('\n[Armors]\n');
        $dataArmors.forEach((armor, index) => {
            if (armor && armor.name) {
                messages.push(removePrefix(armor.name));
                if (armor.description) messages.push(armor.description);
            }
        });

        messages.push('\n[Enemies]\n');
        $dataEnemies.forEach((enemy, index) => {
            if (enemy && enemy.name) {
                messages.push(enemy.name);
            }
        });

        messages.push('\n[States]\n');
        $dataStates.forEach((state, index) => {
            if (state && state.name) {
                messages.push(removePrefix(state.name));
                if (state.message1) messages.push(state.message1);
                if (state.message2) messages.push(state.message2);
                if (state.message3) messages.push(state.message3);
                if (state.message4) messages.push(state.message4);
            }
        });

        messages.push('\n[Types]\n');
        ['weaponTypes', 'armorTypes', 'equipTypes', 'skillTypes'].forEach(typeCategory => {
            if (Array.isArray($dataSystem[typeCategory])) {
                $dataSystem[typeCategory].forEach((type, index) => {
                    if (type) {
                        messages.push(type);
                    }
                });
            }
        });

        messages.push('\n[Terms]\n');
        const terms = $dataSystem.terms;
        Object.keys(terms).forEach(termCategory => {
            if (typeof terms[termCategory] === 'string') {
                messages.push(terms[termCategory]);
            } else if (Array.isArray(terms[termCategory])) {
                terms[termCategory].forEach((term, index) => {
                    messages.push(term);
                });
            }
        });

        messages.push('\n[System Messages]\n');
        if ($dataSystem && $dataSystem.terms && $dataSystem.terms.messages) {
            const messageTypes = Object.keys($dataSystem.terms.messages);
            messageTypes.forEach(key => {
                const messageText = $dataSystem.terms.messages[key];
                if (messageText) {
                    messages.push(messageText);
                }
            });
        }
    }

    function extractMessagesFromMap(mapData, mapId) {
        if (addingMapsName) {
            const mapName = $dataMapInfos[mapId].name;
            messages.push(`\n[${mapName}]\n`);
        }
        if (mapData.events) {
            mapData.events.forEach(event => {
                if (event && event.pages) {
                    event.pages.forEach(page => {
                        if (page.list) {
                            if (extractComments) {
                                let hasAddedCommentHeader = false;
                                page.list.forEach(command => {
                                    if (command.code === 108 || command.code === 408) {
                                        if (!hasAddedCommentHeader) {
                                            messages.push('\n[Comments]\n');
                                            hasAddedCommentHeader = true;
                                        }
                                        messages.push(command.parameters[0]);
                                    }
                                });
                            }

                            let currentMessage = [];
                            let currentSpeaker = '';

                            let currentScrollText = [];

                            page.list.forEach((command, index) => {
                                if (command.code === 405) { // Scroll text
                                    currentScrollText.push(command.parameters[0]);
                                } else if (command.code !== 405 && currentScrollText.length > 0) {
                                    messages.push(currentScrollText.join('\n'));
                                    currentScrollText = [];
                                }

                                if (command.code === 101) { // Text
                                    if (currentMessage.length > 0) {
                                        const fullMessage = currentMessage.join('\n');
                                        messages.push(fullMessage);
                                        if (extractNamesMZ) {
                                            messageData.set(fullMessage, currentSpeaker);
                                        }
                                        currentMessage = [];
                                    }
                                    currentSpeaker = command.parameters[4] || '';
                                } else if (command.code === 401) {
                                    let text = command.parameters[0];
                                    if (excludeNameText) {
                                        const colonIndex = text.indexOf(':');
                                        if (colonIndex !== -1) {
                                            text = text.substring(colonIndex + 1).trim();
                                        }
                                    }
                                    currentMessage.push(text);
                                } else if (command.code === 102) { // Choices
                                    if (currentMessage.length > 0) {
                                        const fullMessage = currentMessage.join('\n');
                                        messages.push(fullMessage);
                                        if (extractNamesMZ) {
                                            messageData.set(fullMessage, currentSpeaker);
                                        }
                                        currentMessage = [];
                                    }
                                    command.parameters[0].forEach(choice => {
                                        messages.push(choice);
                                        if (extractNamesMZ) {
                                            messageData.set(choice, '');
                                        }
                                    });
                                } else if (currentMessage.length > 0) {
                                    const fullMessage = currentMessage.join('\n');
                                    messages.push(fullMessage);
                                    if (extractNamesMZ) {
                                        messageData.set(fullMessage, currentSpeaker);
                                    }
                                    currentMessage = [];
                                    currentSpeaker = '';
                                }
                            });

                            if (currentScrollText.length > 0) {
                                messages.push(currentScrollText.join('\n'));
                            }

                            if (currentMessage.length > 0) {
                                const fullMessage = currentMessage.join('\n');
                                messages.push(fullMessage);
                                if (extractNamesMZ) {
                                    messageData.set(fullMessage, currentSpeaker);
                                }
                            }
                        }
                    });
                }
            });
        }
    }


    function extractTroopMessages() {
        messages.push('\n[TROOPS]\n');
        $dataTroops.forEach((troop, index) => {
            if (troop && troop.pages) {
                troop.pages.forEach((page, pageIndex) => {
                    if (page.list) {
                        let currentMessage = [];
                        let currentSpeaker = '';

                        page.list.forEach(command => {
                            if (command.code === 101) {
                                if (currentMessage.length > 0) {
                                    const fullMessage = currentMessage.join('\n');
                                    messages.push(fullMessage);
                                    if (extractNamesMZ) {
                                        messageData.set(fullMessage, currentSpeaker);
                                    }
                                    currentMessage = [];
                                }
                                currentSpeaker = command.parameters[4] || '';
                            } else if (command.code === 401) {
                                let text = command.parameters[0];
                                if (excludeNameText) {
                                    const colonIndex = text.indexOf(':');
                                    if (colonIndex !== -1) {
                                        text = text.substring(colonIndex + 1).trim();
                                    }
                                }
                                currentMessage.push(text);
                            } else if (command.code === 102) {
                                if (currentMessage.length > 0) {
                                    const fullMessage = currentMessage.join('\n');
                                    messages.push(fullMessage);
                                    if (extractNamesMZ) {
                                        messageData.set(fullMessage, currentSpeaker);
                                    }
                                    currentMessage = [];
                                }
                                command.parameters[0].forEach(choice => {
                                    messages.push(choice);
                                    if (extractNamesMZ) {
                                        messageData.set(choice, '');
                                    }
                                });
                            } else if (currentMessage.length > 0) {
                                const fullMessage = currentMessage.join('\n');
                                messages.push(fullMessage);
                                if (extractNamesMZ) {
                                    messageData.set(fullMessage, currentSpeaker);
                                }
                                currentMessage = [];
                                currentSpeaker = '';
                            }
                        });

                        if (currentMessage.length > 0) {
                            const fullMessage = currentMessage.join('\n');
                            messages.push(fullMessage);
                            if (extractNamesMZ) {
                                messageData.set(fullMessage, currentSpeaker);
                            }
                        }
                    }
                });
            }
        });
    }

    function extractShowPictureMessages() {
        messages.push('\n[SHOW PICTURES]\n');

        function extractFromEventCommands(eventCommands) {
            eventCommands.forEach(command => {
                if (command.code === 231) { // Show Picture
                    let filename = command.parameters[1];
                    if (filename) {
                        if (!filename.toLowerCase().endsWith('.png')) {
                            filename += '.png';
                        }
                        messages.push(filename);
                    }
                }
            });
        }

        for (let mapId = 1; mapId < $dataMapInfos.length; mapId++) {
            if ($dataMapInfos[mapId]) {
                const mapData = JSON.parse(fs.readFileSync(path.join('data', `Map${String(mapId).padStart(3, '0')}.json`), 'utf8'));
                if (mapData.events) {
                    mapData.events.forEach(event => {
                        if (event && event.pages) {
                            event.pages.forEach(page => {
                                if (page.list) {
                                    extractFromEventCommands(page.list);
                                }
                            });
                        }
                    });
                }
            }
        }

        $dataCommonEvents.forEach(commonEvent => {
            if (commonEvent && commonEvent.list) {
                extractFromEventCommands(commonEvent.list);
            }
        });
    }

    function extractPlayMovieMessages() {
        messages.push('\n[PLAY MOVIES]\n');

        function extractFromEventCommands(eventCommands) {
            eventCommands.forEach(command => {
                if (command.code === 261) {
                    let filename = command.parameters[0];
                    if (filename) {
                        if (!filename.toLowerCase().endsWith('.webm')) {
                            filename += '.webm';
                        }
                        messages.push(filename);
                    }
                }
            });
        }

        for (let mapId = 1; mapId < $dataMapInfos.length; mapId++) {
            if ($dataMapInfos[mapId]) {
                const mapData = JSON.parse(fs.readFileSync(path.join('data', `Map${String(mapId).padStart(3, '0')}.json`), 'utf8'));
                if (mapData.events) {
                    mapData.events.forEach(event => {
                        if (event && event.pages) {
                            event.pages.forEach(page => {
                                if (page.list) {
                                    extractFromEventCommands(page.list);
                                }
                            });
                        }
                    });
                }
            }
        }

        $dataCommonEvents.forEach(commonEvent => {
            if (commonEvent && commonEvent.list) {
                extractFromEventCommands(commonEvent.list);
            }
        });
    }

    function extractPluginCommandMessages() {
        messages.push('\n[PLUGIN COMMANDS]\n');

        function isValidContent(content) {
            return /[a-zA-Z]/.test(content) &&
                !['true', 'false'].includes(content.toLowerCase().trim());
        }

        function extractFromEventCommands(eventCommands) {
            eventCommands.forEach(command => {
                if (command.code === 357) { // Plugin Command
                    const params = command.parameters[3] || {};
                    Object.keys(params).forEach(key => {
                        const value = params[key];
                        if (typeof value === 'string') {
                            const keyValueRegex = /"([^"]+)"\s*:\s*"([^"]*)"/g;
                            const regularQuoteRegex = /"([^"]*)"/g;

                            let match;
                            while ((match = keyValueRegex.exec(value)) !== null) {
                                const content = match[2];
                                if (isValidContent(content)) {
                                    messages.push(content);
                                }
                            }

                            if (!keyValueRegex.test(value)) {
                                while ((match = regularQuoteRegex.exec(value)) !== null) {
                                    const content = match[1];
                                    if (isValidContent(content)) {
                                        messages.push(content);
                                    }
                                }
                            }
                        }
                    });
                }
            });
        }

        for (let mapId = 1; mapId < $dataMapInfos.length; mapId++) {
            if ($dataMapInfos[mapId]) {
                const mapData = JSON.parse(fs.readFileSync(path.join('data', `Map${String(mapId).padStart(3, '0')}.json`), 'utf8'));
                if (mapData.events) {
                    mapData.events.forEach(event => {
                        if (event && event.pages) {
                            event.pages.forEach(page => {
                                if (page.list) {
                                    extractFromEventCommands(page.list);
                                }
                            });
                        }
                    });
                }
            }
        }

        $dataCommonEvents.forEach(commonEvent => {
            if (commonEvent && commonEvent.list) {
                extractFromEventCommands(commonEvent.list);
            }
        });
    }

    function updateMessagesCSV() {
        if (!fs || !path) {
            return;
        }
        const filePath = path.join(process.cwd(), 'game_messages.csv');
        let existingContent = [];
        let updatedMessages = new Set();
        let uniqueMessages = new Set();

        if (fs.existsSync(filePath)) {
            const content = fs.readFileSync(filePath, 'utf8');
            existingContent = parseCSV(content);
        }

        const BOM = '\uFEFF';
        const headers = ['Change', 'Excluded'];
        if (extractNamesMZ) {
            headers.push('Name');
        }
        headers.push('Original');
        headers.push(...languageSymbols);

        let csvContent = BOM + headers.join(CSV_SEPARATOR) + '\n';

        const existingTranslations = new Map();
        const existingYepNames = new Map();
        const existingNames = new Map();

        if (existingContent.length > 0) {
            const headers = existingContent[0];
            const changeIndex = headers.indexOf('Change');
            const yepNameIndex = headers.indexOf('Excluded');
            const nameIndex = headers.indexOf('Name');
            const originalIndex = headers.indexOf('Original');

            if (originalIndex !== -1) {
                for (let i = 1; i < existingContent.length; i++) {
                    const row = existingContent[i];
                    if (row.length > originalIndex) {
                        const originalText = row[originalIndex];
                        const translationsStartIndex = originalIndex + 1;
                        const translations = headers.slice(translationsStartIndex).map((lang, index) =>
                            row[translationsStartIndex + index] || '');
                        existingTranslations.set(originalText, translations);

                        if (yepNameIndex !== -1 && row.length > yepNameIndex) {
                            existingYepNames.set(originalText, row[yepNameIndex]);
                        }

                        if (extractNamesMZ && nameIndex !== -1 && row.length > nameIndex) {
                            existingNames.set(originalText, row[nameIndex]);
                        }
                    }
                }
            }
        }

        function processMessage(msg) {
            if (!msg || uniqueMessages.has(msg)) return;
            uniqueMessages.add(msg);

            const { yepName, remainingText } = extractYepName(msg);
            const escapedMsg = escapeCSV(remainingText);
            const escapedYepName = escapeCSV(yepName);
            let line = ['', escapedYepName];

            if (extractNamesMZ) {
                const speakerName = messageData.get(msg) || '';
                const escapedSpeakerName = escapeCSV(speakerName);
                line.push(escapedSpeakerName);
            }

            line.push(escapedMsg);

            if (existingTranslations.has(remainingText)) {
                line[0] = '';
                line = line.concat(existingTranslations.get(remainingText).map(t => escapeCSV(t)));
            } else {
                line[0] = 'NEW';
                line = line.concat(Array(languageSymbols.length).fill(''));
            }

            csvContent += line.join(CSV_SEPARATOR) + '\n';
            updatedMessages.add(remainingText);
        }

        messages.forEach(processMessage);

        if (addingMapsName) {
            const mapNames = messages.filter(msg => msg && msg.startsWith('[') && msg !== '[DATABASE ENTRIES]' && !msg.startsWith('[Choice]'));
            mapNames.forEach(processMessage);
        }

        if (messages.includes('[Common Events]')) {
            csvContent += `,,[Common Events]${CSV_SEPARATOR.repeat(languageSymbols.length + 1)}\n`;
            const commonEventMessages = messages.slice(messages.indexOf('[Common Events]') + 1);
            commonEventMessages.forEach(processMessage);
        }

        if (extractVariableText && messages.includes('[VARIABLES]')) {
            csvContent += `,,[VARIABLES]${CSV_SEPARATOR.repeat(languageSymbols.length + 1)}\n`;
            const variableMessages = messages.slice(messages.indexOf('[VARIABLES]') + 1);
            variableMessages.forEach(processMessage);
        }

        if (extractDatabaseEntries) {
            csvContent += `,,[DATABASE ENTRIES]${CSV_SEPARATOR.repeat(languageSymbols.length + 1)}\n`;
            const databaseContentIndex = messages.findIndex(msg => msg === '[DATABASE ENTRIES]');
            const databaseContent = databaseContentIndex !== -1 ? messages.slice(databaseContentIndex + 1) : [];

            let currentCategory = '';
            databaseContent.forEach((item) => {
                if (!item) return;
                if (/^\[.+\]$/.test(item)) {
                    currentCategory = item;
                    csvContent += `,,${escapeCSV(item)}${CSV_SEPARATOR.repeat(languageSymbols.length + 1)}\n`;
                } else if (item.trim() !== '' && item !== '[DATABASE ENTRIES]') {
                    processMessage(item);
                }
            });
        }

        csvContent += `,,${escapeCSV('[GARBAGE AND \nCUSTOM LINES]')}${CSV_SEPARATOR.repeat(languageSymbols.length + 1)}\n`;

        let customLines = '';
        existingContent.slice(1).forEach(row => {
            if (!row || row.length < 3) return;

            const headers = existingContent[0];
            const changeIndex = headers.indexOf('Change');
            const excludedIndex = headers.indexOf('Excluded');
            const nameIndex = headers.indexOf('Name');
            const originalIndex = headers.indexOf('Original');

            const originalText = row[originalIndex];

            if (!updatedMessages.has(originalText) &&
                originalText !== '[GARBAGE AND \nCUSTOM LINES]' &&
                originalText !== '[DATABASE ENTRIES]') {

                let newRow = [];

                newRow.push(row[changeIndex] || '');
                newRow.push(row[excludedIndex] || '');

                if (extractNamesMZ && nameIndex !== -1) {
                    newRow.push(row[nameIndex] || '');
                }

                newRow.push(originalText);
                const translations = languageSymbols.map(symbol => {
                    const langIndex = headers.indexOf(symbol);
                    return langIndex !== -1 ? row[langIndex] || '' : '';
                });
                newRow = newRow.concat(translations);

                customLines += newRow.map(escapeCSV).join(CSV_SEPARATOR) + '\n';
            }
        });

        csvContent += customLines;

        if (addGibberishLines) {
            csvContent += `,,[GIBBERISH LINES FOR PERFORMANCE TESTING]${CSV_SEPARATOR.repeat(languageSymbols.length + 1)}\n`;
            for (let i = 0; i < 80000; i++) {
                const gibberish = generateGibberish(50);
                csvContent += `NEW,,${escapeCSV(gibberish)}${CSV_SEPARATOR.repeat(languageSymbols.length + 1)}\n`;
            }
        }

        // Write the CSV file
        fs.writeFile(filePath, csvContent, { encoding: 'utf8' }, (err) => {
            if (err) {
                //console.error('Error writing CSV file:', err);
            } else {
                //console.log('Messages updated in:', filePath);
            }
        });
    }

    function escapeCSV(text) {
        if (typeof text !== 'string') {
            return text;
        }
        if (text.includes('"') || text.includes(CSV_SEPARATOR) || text.includes('\n') || text.includes('\r')) {
            return '"' + text.replace(/"/g, '""') + '"';
        }
        return text;
    }

    function parseCSV(text) {
        const separator = detectCSVSeparatorFromContent(text);
        const lines = [];
        let currentLine = [];
        let currentField = '';
        let withinQuotes = false;

        for (let i = 0; i < text.length; i++) {
            const char = text[i];
            if (char === '"') {
                if (withinQuotes && text[i + 1] === '"') {
                    currentField += '"';
                    i++;
                } else {
                    withinQuotes = !withinQuotes;
                }
            } else if (char === separator && !withinQuotes) {
                currentLine.push(currentField);
                currentField = '';
            } else if ((char === '\n' || char === '\r') && !withinQuotes) {
                if (char === '\r' && text[i + 1] === '\n') {
                    i++;
                }
                currentLine.push(currentField);
                lines.push(currentLine);
                currentLine = [];
                currentField = '';
            } else {
                currentField += char;
            }
        }

        if (currentField) {
            currentLine.push(currentField);
        }
        if (currentLine.length > 0) {
            lines.push(currentLine);
        }

        return lines;
    }

    let wordTranslations = {};

    function loadTranslations(lang) {
        let filePath = 'game_messages.csv';
        translations = {};
        wordTranslations = {};
        wordTranslationRegexes.clear();
        translationCache = {};

        let content = '';
        if (typeof require === 'function') {
            // Desktop
            if (fs.existsSync(filePath)) {
                content = fs.readFileSync(filePath, 'utf8');
            } else {
                // www folder (for deployed projects)
                filePath = path.join('www', filePath);
                if (fs.existsSync(filePath)) {
                    content = fs.readFileSync(filePath, 'utf8');
                } else {
                    console.error('Translation file not found in root or www folder');
                    return;
                }
            }
        } else {
            // Mobile
            const xhr = new XMLHttpRequest();
            xhr.open('GET', filePath, false);
            xhr.overrideMimeType('text/plain; charset=utf-8');
            xhr.send();
            if (xhr.status === 200) {
                content = xhr.responseText;
            } else {
                throw new Error('HTTP status: ' + xhr.status);
            }
        }

        if (!content) {
            return;
        }

        const lines = parseCSV(content);
        if (lines.length > 0) {
            const headers = lines[0];
            const yepNameIndex = headers.indexOf('Excluded');
            const originalIndex = headers.indexOf('Original');
            const langIndex = headers.indexOf(lang);

            availableLanguages = [];

            // See which languages have non-empty columns
            languages.forEach(langObj => {
                const langColIndex = headers.indexOf(langObj.Symbol);
                if (langColIndex !== -1) {
                    let hasTranslation = false;
                    for (let j = 1; j < lines.length; j++) {
                        if (lines[j][langColIndex] && lines[j][langColIndex].trim() !== '') {
                            hasTranslation = true;
                            break;
                        }
                    }
                    if (hasTranslation) {
                        availableLanguages.push(langObj.Symbol);
                    }
                }
            });

            // If the current language is not available, switch to the first available language
            if (!availableLanguages.includes(currentLanguage)) {
                currentLanguage = availableLanguages[0] || defaultLanguage;
                ConfigManager.language = currentLanguage;
            }
            //////////////////////////////////////////////////////

            if (originalIndex !== -1 && langIndex !== -1 && yepNameIndex !== -1) {
                for (let i = 1; i < lines.length; i++) {
                    const columns = lines[i];
                    if (columns.length > Math.max(originalIndex, langIndex, yepNameIndex)) {
                        const originalText = columns[originalIndex];
                        const translatedText = columns[langIndex].replace(/\n/g, '{{LINEBREAK}}');
                        const yepName = columns[yepNameIndex];

                        if (originalText && translatedText) {
                            translations[originalText] = translatedText;

                            if (yepName) {
                                translations[yepName + originalText] = translatedText;
                                wordTranslations[yepName] = translatedText;
                            }
                        }
                    }
                }
            }
        }

        originalTexts.length = 0;
        Object.keys(translations)
            .sort((a, b) => b.length - a.length)
            .forEach(key => originalTexts.push(key));
    }

    Game_Message.prototype.messageBuffer = [];

    const _Game_Message_add = Game_Message.prototype.add;
    Game_Message.prototype.add = function (text) {
        if (this._scrollMode || ($gameParty.inBattle() && !$gameMessage._showFast)) {
            this._texts.push(text);
        } else {
            this.messageBuffer.push(text);
        }
    };

    const _Scene_Map_start = Scene_Map.prototype.start;
    Scene_Map.prototype.start = function () {
        if ($gameMessage.messageBuffer) {
            $gameMessage.messageBuffer = [];
        }
        _Scene_Map_start.call(this);
    };

    const _Game_Map_setup = Game_Map.prototype.setup;
    Game_Map.prototype.setup = function (mapId) {
        if ($gameMessage.messageBuffer) {
            $gameMessage.messageBuffer = [];
        }
        _Game_Map_setup.call(this, mapId);
    };

    // Play Movies
    const _Game_Interpreter_command261 = Game_Interpreter.prototype.command261;
    Game_Interpreter.prototype.command261 = function () {
        let filename = Utils.RPGMAKER_NAME === "MV" ? this._params[0] : arguments[0][0];
        const originalHadWebm = filename.toLowerCase().endsWith('.webm');

        if (!originalHadWebm) {
            filename += '.webm';
        }

        let translatedName = translations[filename] || filename;

        if (!originalHadWebm && translatedName.toLowerCase().endsWith('.webm')) {
            translatedName = translatedName.slice(0, -5);
        }

        if (Utils.RPGMAKER_NAME === "MV") {
            this._params[0] = translatedName;
            return _Game_Interpreter_command261.call(this);
        } else {
            arguments[0][0] = translatedName;
            return _Game_Interpreter_command261.call(this, arguments[0]);
        }
    };

    // Script Call
    const _Game_Interpreter_command355 = Game_Interpreter.prototype.command355;
    Game_Interpreter.prototype.command355 = function (params) {
        const scriptText = Utils.RPGMAKER_NAME === 'MZ' ? params[0] : this._params[0];

        let allLines = [scriptText];
        let nextIndex = this._index + 1;
        while (this._list[nextIndex] && this._list[nextIndex].code === 655) {
            allLines.push(Utils.RPGMAKER_NAME === 'MZ' ?
                this._list[nextIndex].parameters[0] :
                this._list[nextIndex].parameters[0]);
            nextIndex++;
        }

        const processedLines = allLines.map(line => {
            return line.replace(/"([^"]*)"/g, (match, p1) => {
                const translated = translateText(p1);
                return `"${translated}"`;
            });
        });

        if (Utils.RPGMAKER_NAME === 'MZ') {
            params[0] = processedLines[0];
            for (let i = 1; i < processedLines.length; i++) {
                this._list[this._index + i].parameters[0] = processedLines[i];
            }
            return _Game_Interpreter_command355.call(this, params);
        } else {
            this._params[0] = processedLines[0];
            for (let i = 1; i < processedLines.length; i++) {
                this._list[this._index + i].parameters[0] = processedLines[i];
            }
            return _Game_Interpreter_command355.call(this);
        }
    };

    // Single continuation lines (in case they're called directly)
    const _Game_Interpreter_command655 = Game_Interpreter.prototype.command655;
    Game_Interpreter.prototype.command655 = function (params) {
        const scriptText = Utils.RPGMAKER_NAME === 'MZ' ? params[0] : this._params[0];
        const script = scriptText.replace(/"([^"]*)"/g, (match, p1) => {
            const translated = translateText(p1);
            return `"${translated}"`;
        });
        if (Utils.RPGMAKER_NAME === 'MZ') {
            params[0] = script;
            return _Game_Interpreter_command655.call(this, params);
        } else {
            this._params[0] = script;
            return _Game_Interpreter_command655.call(this);
        }
    };

    // Fix for script call showMessage add
    const _Game_Interpreter_command355_orig = Game_Interpreter.prototype.command355;
    Game_Interpreter.prototype.command355 = function () {
        const result = _Game_Interpreter_command355_orig.apply(this, arguments);
        if ($gameMessage.messageBuffer && $gameMessage.messageBuffer.length > 0) {
            $gameMessage.processMessageBuffer();
        }
        return result;
    };

    const _Game_Interpreter_command655_orig = Game_Interpreter.prototype.command655;
    Game_Interpreter.prototype.command655 = function () {
        const result = _Game_Interpreter_command655_orig.apply(this, arguments);
        if ($gameMessage.messageBuffer && $gameMessage.messageBuffer.length > 0) {
            $gameMessage.processMessageBuffer();
        }
        return result;
    };

    // SCROLL TEXT_____________________________________
    const _Window_ScrollText_initialize = Window_ScrollText.prototype.initialize;
    Window_ScrollText.prototype.initialize = function () {
        if (Utils.RPGMAKER_NAME === "MV") {
            _Window_ScrollText_initialize.call(this);
        } else {
            const rect = new Rectangle(0, 0, Graphics.boxWidth, Graphics.boxHeight);
            _Window_ScrollText_initialize.call(this, rect);
        }
        this._originalLines = 0;
    };

    const _Window_ScrollText_startMessage = Window_ScrollText.prototype.startMessage;
    Window_ScrollText.prototype.startMessage = function () {
        this._originalLines = $gameMessage._texts.length;  // Store original line count
        let fullText = $gameMessage._texts.join('\n');     // Join all text
        let translatedText = translateText(fullText);      // Translate the full text
        $gameMessage._texts = translatedText.split('\n');  // Split back into lines

        _Window_ScrollText_startMessage.call(this);
    };

    //___________________________________________________
    Game_Message.prototype.processMessageBuffer = Game_Message.prototype.processMessageBuffer || function () {
        if (!this.messageBuffer.length) return;

        const fullMessage = this.messageBuffer.join('\n');
        let processedText = fullMessage;

        if (Imported['YEP_MessageCore'] && $gameSystem.wordWrap()) {
            processedText = '<WordWrap>' + processedText;
        }

        const translatedText = translateText(processedText);
        this.messageBuffer = [];

        translatedText.split('\n').forEach(line => {
            _Game_Message_add.call(this, line);
        });
    };

    if (Utils.RPGMAKER_NAME === "MV") {
        if (Imported['YEP_MessageCore']) {
            Game_Interpreter.prototype.command101 = function () {
                if (!$gameMessage.isBusy()) {
                    $gameMessage.setFaceImage(this._params[0], this._params[1]);
                    $gameMessage.setBackground(this._params[2]);
                    $gameMessage.setPositionType(this._params[3]);

                    if (this._params[4]) {
                        $gameMessage.addText(this._params[4]);
                    }

                    while (this.isContinueMessageString()) {
                        this._index++;
                        if (this._list[this._index].code === 401) {
                            $gameMessage.add(this.currentCommand().parameters[0]);
                        }
                        if ($gameMessage._texts.length >= $gameSystem.messageRows()) break;
                    }

                    $gameMessage.processMessageBuffer();

                    switch (this.nextEventCode()) {
                        case 102:
                            this._index++;
                            this.setupChoices(this.currentCommand().parameters);
                            break;
                        case 103:
                            this._index++;
                            this.setupNumInput(this.currentCommand().parameters);
                            break;
                        case 104:
                            this._index++;
                            this.setupItemChoice(this.currentCommand().parameters);
                            break;
                    }
                    this._index++;
                    this.setWaitMode('message');
                }
                return false;
            };
        } else {
            const _Game_Interpreter_command101 = Game_Interpreter.prototype.command101;
            Game_Interpreter.prototype.command101 = function () {
                const result = _Game_Interpreter_command101.call(this);
                $gameMessage.processMessageBuffer();
                return result;
            };
        }
    } else if (Utils.RPGMAKER_NAME === "MZ") {
        const _Game_Interpreter_command101 = Game_Interpreter.prototype.command101;
        Game_Interpreter.prototype.command101 = function (params) {
            const result = _Game_Interpreter_command101.call(this, params);
            $gameMessage.processMessageBuffer();
            return result;
        };
    }

    const _Game_Variables_setValue = Game_Variables.prototype.setValue;
    Game_Variables.prototype.setValue = function (variableId, value) {
        if (typeof value === 'string') {
            const translationKey = `__VARIABLE__${variableId}__`;
            if (translations.hasOwnProperty(translationKey)) {
                value = translations[translationKey];
            } else {
                value = translateText(value);
            }
        }
        _Game_Variables_setValue.call(this, variableId, value);
    };

    if (Utils.RPGMAKER_NAME === "MZ") {
    const _Bitmap_drawText = Bitmap.prototype.drawText;
    Bitmap.prototype.drawText = function (text, x, y, maxWidth, lineHeight, align) {
        const translatedText = translateText(String(text));
        _Bitmap_drawText.call(this, translatedText, x, y, maxWidth, lineHeight, align);
    };}

    // Override drawTextEx method so other plugins will be affected, hopefully
    const _Window_Base_drawTextEx = Window_Base.prototype.drawTextEx;
    Window_Base.prototype.drawTextEx = function (text, x, y, width) {
        if (text) {
            text = Hendrix_Localization(text);
            const originalHendrixLocalization = window.Hendrix_Localization;
            window.Hendrix_Localization = function (innerText) {
                return innerText;
            };
            const result = _Window_Base_drawTextEx.call(this, text, x, y, width);
            window.Hendrix_Localization = originalHendrixLocalization;
            return result;
        } else {
            return 0;
        }
    };

    if (Utils.RPGMAKER_NAME === "MZ") {
        const _Game_Message_setChoices = Game_Message.prototype.setChoices;
        Game_Message.prototype.setChoices = function (choices, defaultType, cancelType) {
            const translatedChoices = choices.map(choice => {
                const escapeCodes = choice.match(/<.*?>/g) || [];
                let textContent = choice;
                escapeCodes.forEach(code => {
                    textContent = textContent.replace(code, '');
                });
                let translatedText = translations[textContent] || textContent;

                escapeCodes.forEach(code => {
                    const originalIndex = choice.indexOf(code);
                    if (originalIndex === 0) {
                        translatedText = code + translatedText;
                    } else {
                        translatedText = translatedText + code;
                    }
                });

                return translatedText;
            });

            _Game_Message_setChoices.call(this, translatedChoices, defaultType, cancelType);
        };

        const _Window_NameBox_windowWidth = Window_NameBox.prototype.windowWidth;
        Window_NameBox.prototype.windowWidth = function () {
            if (this._name) {
                const cleanName = (translations[this._name] || this._name).replace(/<[^>]*>/g, '');
                return Math.ceil(this.textWidth(cleanName) + this.padding * 2 + this.itemPadding() * 2);
            } else {
                return _Window_NameBox_windowWidth.call(this);
            }
        };
    }

    // Fix choice issue: Escape code from database doesn't translate (YEP Message Core)
    if (Utils.RPGMAKER_NAME === "MV") {
        if (Imported['YEP_MessageCore']) {
            const _Game_Message_setChoices = Game_Message.prototype.setChoices;
            Game_Message.prototype.setChoices = function (choices, defaultType, cancelType) {
                const translatedChoices = choices.map(choice => {
                    // Process escape codes like \ii[x] etc
                    const window = SceneManager._scene._messageWindow;
                    if (window) {
                        choice = window.convertEscapeCharacters(choice);
                    }
                    return Hendrix_Localization(choice);
                });
                _Game_Message_setChoices.call(this, translatedChoices, defaultType, cancelType);
            };
        }
    }

    // Fix Battle Message Troop doesn't translate whole paragraph
    const _Window_Message_startMessage = Window_Message.prototype.startMessage;
    Window_Message.prototype.startMessage = function () {
        if ($gameParty.inBattle()) {
            const allText = $gameMessage._texts.join('\n');
            const translatedText = translateText(allText);
            $gameMessage._texts = translatedText.split('\n');
        }
        _Window_Message_startMessage.call(this);
    };

    ConfigManager.language = defaultLanguage;

    const _ConfigManager_makeData = ConfigManager.makeData;
    ConfigManager.makeData = function () {
        const config = _ConfigManager_makeData.call(this);
        config.language = this.language;
        return config;
    };

    const _ConfigManager_applyData = ConfigManager.applyData;
    ConfigManager.applyData = function (config) {
        _ConfigManager_applyData.call(this, config);
        this.language = config.language || defaultLanguage;
        currentLanguage = this.language;
        loadTranslations(currentLanguage);
        applyLanguageSettings();
    };

    const _Window_Options_addGeneralOptions = Window_Options.prototype.addGeneralOptions;
    Window_Options.prototype.addGeneralOptions = function () {
        _Window_Options_addGeneralOptions.call(this);
        this.addCommand('Language', 'language');
    };

    const _Window_Options_statusText = Window_Options.prototype.statusText;
    Window_Options.prototype.statusText = function (index) {
        const symbol = this.commandSymbol(index);
        if (symbol === 'language') {
            const langObj = languages.find(lang => lang.Symbol === ConfigManager.language);
            return langObj ? langObj.Name : '';
        } else {
            return _Window_Options_statusText.call(this, index);
        }
    };

    const _Window_Options_processOk = Window_Options.prototype.processOk;
    Window_Options.prototype.processOk = function () {
        const index = this.index();
        const symbol = this.commandSymbol(index);
        if (symbol === 'language') {
            this.changeLanguage(1);
        } else {
            _Window_Options_processOk.call(this);
        }
    };

    const _Window_Options_cursorRight = Window_Options.prototype.cursorRight;
    Window_Options.prototype.cursorRight = function (wrap) {
        const index = this.index();
        const symbol = this.commandSymbol(index);
        if (symbol === 'language') {
            this.changeLanguage(1);
        } else {
            _Window_Options_cursorRight.call(this, wrap);
        }
    };

    const _Window_Options_cursorLeft = Window_Options.prototype.cursorLeft;
    Window_Options.prototype.cursorLeft = function (wrap) {
        const index = this.index();
        const symbol = this.commandSymbol(index);
        if (symbol === 'language') {
            this.changeLanguage(-1);
        } else {
            _Window_Options_cursorLeft.call(this, wrap);
        }
    };

    Window_Options.prototype.changeLanguage = function (direction) {
        const currentIndex = availableLanguages.indexOf(ConfigManager.language);

        let nextIndex = (currentIndex + direction + availableLanguages.length) % availableLanguages.length;

        ConfigManager.language = availableLanguages[nextIndex];
        currentLanguage = ConfigManager.language;

        this.redrawItem(this.findSymbol('language'));
        loadTranslations(currentLanguage);
        applyLanguageSettings();

        if (Imported.YEP_CommonEventMenu) { DataManager.processCEMNotetags1($dataCommonEvents); }

        if (autoRefreshPictures) {
            refreshAllPictures();
        }
        if (SceneManager._scene && SceneManager._scene.refreshAllWindows) {
            SceneManager._scene.refreshAllWindows();
        }
        SoundManager.playCursor();
    };

    Scene_Base.prototype.refreshAllWindows = function () {
        if (this._windowLayer) {
            this._windowLayer.children.forEach(child => {
                if (child instanceof Window_Base) {
                    if (child.contents) {
                        child.contents.clear();
                    }
                    if (typeof child.createContents === 'function') {
                        child.createContents();
                    }
                    if (typeof child.refresh === 'function' &&
                        !(child instanceof Window_Message) &&
                        !(child instanceof Window_ChoiceList)) {
                        child.refresh();
                    }
                    if (typeof child.updatePadding === 'function') {
                        child.updatePadding();
                    }
                    if (typeof child.resetFontSettings === 'function') {
                        child.resetFontSettings();
                    }
                }
            });
        }
    };

    function refreshAllPictures() {
        if ($gameScreen && $gameScreen._pictures) {
            for (let i = 1; i <= $gameScreen.maxPictures(); i++) {
                const picture = $gameScreen.picture(i);
                if (picture) {
                    const originalName = picture.originalName;
                    let newName = originalName;
                    if (!newName.toLowerCase().endsWith('.png')) {
                        newName += '.png';
                    }
                    newName = translatePictureName(newName);
                    if (!originalName.toLowerCase().endsWith('.png') && newName.toLowerCase().endsWith('.png')) {
                        newName = newName.slice(0, -4);
                    }
                    picture.changeName(newName);
                }
            }
        }
    }

    Game_Picture.prototype.changeName = function (name) {
        if (this._name !== name) {
            this._name = name;
            this.initTarget();
        }
    };

    Object.defineProperty(Game_Picture.prototype, 'originalName', {
        get: function () {
            return this._originalName || this._name;
        },
        set: function (value) {
            this._originalName = value;
        }
    });


    function translatePictureName(name) {
        if (translations.hasOwnProperty(name)) {
            return translations[name];
        }
        return name;
    }

    function translateText(text) {
        if (typeof text !== 'string') {
            return text;
        }

        if (useTranslationCache && translationCache.hasOwnProperty(text)) {
            return translationCache[text];
        }

        if (text.toLowerCase().endsWith('.png')) {
            return text;
        }

        let translatedText = text;

        // First pass: full sentence and partial matches
        const { yepName, remainingText } = extractYepName(translatedText);

        if (translations.hasOwnProperty(yepName + remainingText)) {
            translatedText = yepName + translations[yepName + remainingText];
        } else if (translations.hasOwnProperty(remainingText)) {
            translatedText = yepName + translations[remainingText];
        } else {
            const lines = remainingText.split('\n');
            const translatedLines = lines.map(line => {
                if (translations.hasOwnProperty(line.trim())) {
                    return translations[line.trim()];
                } else if (partialMatching) {
                    for (let i = 0, len = originalTexts.length; i < len; i++) {
                        const originalText = originalTexts[i];
                        if (line.includes(originalText)) {
                            return line.replace(originalText, translations[originalText] || originalText);
                        }
                    }
                }
                return line;
            });

            translatedLines[0] = yepName + translatedLines[0];
            translatedText = translatedLines.join('\n');
        }

        // Second pass: word-level translations
        Object.keys(wordTranslations).forEach(word => {
            let regex = wordTranslationRegexes.get(word);
            if (!regex) {
                regex = new RegExp(`${wordBoundaryRegex.source}${escapeRegExp(word)}${wordBoundaryRegex.source}`, 'g');
                wordTranslationRegexes.set(word, regex);
            }
            translatedText = translatedText.replace(regex, wordTranslations[word]);
        });

        translatedText = translatedText.replace(/{{LINEBREAK}}/g, '\n');

        if (useTranslationCache) {
            translationCache[text] = translatedText;
        }
        return translatedText;
    }

    // Override drawText method
    //const _Window_Base_drawText = Window_Base.prototype.drawText;
    //Window_Base.prototype.drawText = function (text, x, y, maxWidth, align) {
    //    const translatedText = Hendrix_Localization((text));
    //    _Window_Base_drawText.call(this, translatedText, x, y, maxWidth, align);
    //};

    const _Game_Actor_profile = Game_Actor.prototype.profile;
    Game_Actor.prototype.profile = function () {
        return translateText(_Game_Actor_profile.call(this));
    };

    // Translate Show Pictures and Title background file
    const _Game_Screen_showPicture = Game_Screen.prototype.showPicture;
    Game_Screen.prototype.showPicture = function (pictureId, name, origin, x, y, scaleX, scaleY, opacity, blendMode) {
        let translatedName = name;
        let originalHadPng = name.toLowerCase().endsWith('.png');

        if (!originalHadPng) {
            translatedName += '.png';
        }
        translatedName = translatePictureName(translatedName);
        if (!originalHadPng && translatedName.toLowerCase().endsWith('.png')) {
            translatedName = translatedName.slice(0, -4);
        }

        _Game_Screen_showPicture.call(this, pictureId, translatedName, origin, x, y, scaleX, scaleY, opacity, blendMode);

        const picture = this.picture(pictureId);
        if (picture) {
            picture._originalName = name;
        }

    };

    const _Scene_Title_drawGameTitle = Scene_Title.prototype.drawGameTitle;
    Scene_Title.prototype.drawGameTitle = function () {
        if (DataManager._originalGameTitle && $dataSystem) {
            const translatedTitle = translateText(DataManager._originalGameTitle);
            $dataSystem.gameTitle = translatedTitle;
        }
        _Scene_Title_drawGameTitle.call(this);
    };

    const _Scene_Title_createBackground = Scene_Title.prototype.createBackground;
    Scene_Title.prototype.createBackground = function () {
        if (!originalTitle1) originalTitle1 = $dataSystem.title1Name;
        if (!originalTitle2) originalTitle2 = $dataSystem.title2Name;

        if (originalTitle1) {
            const title1File = originalTitle1 + '.png';
            const translatedTitle1 = translatePictureName(title1File);
            if (translatedTitle1 !== title1File) {
                $dataSystem.title1Name = translatedTitle1.replace('.png', '');
            }
        }

        if (originalTitle2) {
            const title2File = originalTitle2 + '.png';
            const translatedTitle2 = translatePictureName(title2File);
            if (translatedTitle2 !== title2File) {
                $dataSystem.title2Name = translatedTitle2.replace('.png', '');
            }
        }

        _Scene_Title_createBackground.call(this);
    };

    if (extractMapDisplayNames) {
        Window_MapName.prototype.refresh = function () {
            this.contents.clear();
            if ($gameMap.displayName()) {
                const translatedName = translateText($gameMap.displayName());
                if (Utils.RPGMAKER_NAME === "MZ") {
                    const width = this.innerWidth;
                    this.drawBackground(0, 0, width, this.lineHeight());
                    this.drawText(translatedName, 0, 0, width, 'center');
                } else {
                    const width = this.contentsWidth();
                    this.drawBackground(0, 0, width, this.lineHeight());
                    this.drawText(translatedName, 0, 0, width, 'center');
                }
            }
        };

        const _Game_Map_displayName = Game_Map.prototype.displayName;
        Game_Map.prototype.displayName = function () {
            const originalName = _Game_Map_displayName.call(this);
            return translateText(originalName);
        };

        const _DataManager_makeSavefileInfo = DataManager.makeSavefileInfo;
        DataManager.makeSavefileInfo = function () {
            const info = _DataManager_makeSavefileInfo.call(this);
            if (info.location) {
                info.location = translateText(info.location);
            }
            return info;
        };

        const _Scene_Menu_mapNameWindow = Scene_Menu.prototype.createMapNameWindow || function () { };
        Scene_Menu.prototype.createMapNameWindow = function () {
            _Scene_Menu_mapNameWindow.call(this);
            if (this._mapNameWindow && this._mapNameWindow.refresh) {
                this._mapNameWindow.refresh();
            }
        };
    }
    //---------------------------------------------------------

    Window_Help.prototype.refresh = function () {
        this.contents.clear();
        if (this._text) {
            const translatedText = translateText(this._text);
            if (this.baseTextRect) {
                // MZ
                const rect = this.baseTextRect();
                this.drawTextEx(translatedText, rect.x, rect.y, rect.width);
            } else {
                // MV
                const rect = {
                    x: this.textPadding(),
                    y: 0,
                    width: this.contents.width - this.textPadding() * 2,
                    height: this.contents.height
                };
                this.drawTextEx(translatedText, rect.x, rect.y, rect.width);
            }
        }
    };

    if (!Window_Base.prototype.calcTextWidth) {
        Window_Base.prototype.calcTextWidth = function (text) {
            var tempText = text.split('\n');
            var maxWidth = 0;
            for (var i = 0; i < tempText.length; i++) {
                var textWidth = this.textWidth(tempText[i]);
                if (maxWidth < textWidth) {
                    maxWidth = textWidth;
                }
            }
            return maxWidth;
        };
    }

    const _Window_Base_processEscapeCharacter = Window_Base.prototype.processEscapeCharacter;
    Window_Base.prototype.processEscapeCharacter = function (code, textState) {
        if (code === 'T') {
            const endIndex = textState.text.indexOf(']', textState.index);
            const textToTranslate = textState.text.substring(textState.index, endIndex);
            const translatedText = translateText(textToTranslate);
            textState.text = textState.text.replace(`T[${textToTranslate}]`, translatedText);
            textState.index = endIndex + 1;
        } else {
            _Window_Base_processEscapeCharacter.call(this, code, textState);
        }
    };

    window.Hendrix_Localization = function (text) {
        if (typeof translateText === 'function') {
            return translateText(text);
        }
        return text;
    };

    if (Imported['Galv_MessageStyles']) {
        Window_Message.prototype.changeWindowDimensions = function () {
            if (this.pTarget != null) {
                var w = 10;
                var h = 0;

                if (Imported.Galv_MessageBusts) {
                    if ($gameMessage.bustPos == 1) {
                        var faceoffset = 0;
                    } else {
                        var faceoffset = Galv.MB.w;
                    };
                } else {
                    var faceoffset = Window_Base._faceWidth + 25;
                };
                var xO = $gameMessage._faceName ? faceoffset : 0;
                xO += Galv.Mstyle.padding[1] + Galv.Mstyle.padding[3];

                this.resetFontSettings();
                for (var i = 0; i < $gameMessage._texts.length; i++) {
                    var lineWidth = this.testWidthEx($gameMessage._texts[i]) + this.standardPadding() * 2 + xO;
                    if (w < lineWidth) w = lineWidth;

                };
                this.resetFontSettings();
                this.width = Math.min(Graphics.boxWidth, w);

                var minFaceHeight = 0;
                if ($gameMessage._faceName) {
                    w += 15;
                    if (Imported.Galv_MessageBusts) {
                        if ($gameMessage.bustPos == 1) w += Galv.MB.w;
                        minFaceHeight = 0;
                    } else {
                        minFaceHeight = Window_Base._faceHeight + this.standardPadding() * 2;
                    };
                };

                var textState = { index: 0 };
                textState.text = this.convertEscapeCharacters($gameMessage.allText());
                var allLineHeight = this.calcTextHeight(textState, true);
                var height = allLineHeight + this.standardPadding() * 2;
                var minHeight = this.fittingHeight(0);
                this.height = Math.max(height, minHeight, minFaceHeight);
                this.yOffset = -Galv.Mstyle.yOffet - this.height;

            } else {
                this.yOffset = 0;
                this.width = this.windowWidth();
                this.height = Galv.Mstyle.Window_Message_windowHeight.call(this);
                this.x = (Graphics.boxWidth - this.width) / 2;
            };
        };
    }

    if (Imported.YEP_CommonEventMenu) {
        DataManager.processCEMNotetags1 = function (group) {
            for (var n = 1; n < group.length; n++) {
                var obj = group[n];
                var notedata = this.convertCommentsToText(obj);

                obj.iconIndex = Yanfly.Param.CEMIcon;
                obj.description = Hendrix_Localization(Yanfly.Param.CEMHelpDescription);
                obj.picture = '';
                obj.menuSettings = {
                    name: obj.name,
                    subtext: Hendrix_Localization(Yanfly.Param.CEMSubtext),
                    enabled: 'enabled = true',
                    show: 'visible = true'
                };
                var evalMode = 'none';

                for (var i = 0; i < notedata.length; i++) {
                    var line = notedata[i];
                    if (line.match(/<MENU NAME:[ ](.*)>/i)) {
                        var translatedLine = Hendrix_Localization(line);
                        if (translatedLine.match(/<MENU NAME:[ ](.*)>/i)) {
                            obj.menuSettings.name = String(RegExp.$1);
                        } else {
                            var menuName = String(RegExp.$1);
                            obj.menuSettings.name = Hendrix_Localization(menuName);
                        }
                    } else if (line.match(/<ICON:[ ](\d+)>/i)) {
                        obj.iconIndex = parseInt(RegExp.$1);
                    } else if (line.match(/<PICTURE:[ ](.*)>/i)) {
                        obj.picture = String(RegExp.$1);
                    } else if (line.match(/<HELP DESCRIPTION>/i)) {
                        evalMode = 'help description';
                        obj.description = '';
                    } else if (line.match(/<\/HELP DESCRIPTION>/i)) {
                        evalMode = 'none';
                        obj.description = Hendrix_Localization(obj.description);
                    } else if (evalMode === 'help description') {
                        obj.description += line + '\n';
                    } else if (line.match(/<SUBTEXT>/i)) {
                        evalMode = 'subtext';
                        obj.menuSettings.subtext = '';
                    } else if (line.match(/<\/SUBTEXT>/i)) {
                        evalMode = 'none';
                        obj.menuSettings.subtext = Hendrix_Localization(obj.menuSettings.subtext);
                    } else if (evalMode === 'subtext') {
                        obj.menuSettings.subtext += line + '\n';
                    } else if (line.match(/<MENU ENABLE EVAL>/i)) {
                        evalMode = 'menu enable eval';
                        obj.menuSettings.enabled = '';
                    } else if (line.match(/<\/MENU ENABLE EVAL>/i)) {
                        evalMode = 'none';
                    } else if (evalMode === 'menu enable eval') {
                        obj.menuSettings.enabled += line + '\n';
                    } else if (line.match(/<MENU VISIBLE EVAL>/i)) {
                        evalMode = 'menu visible eval';
                        obj.menuSettings.show = '';
                    } else if (line.match(/<\/MENU VISIBLE EVAL>/i)) {
                        evalMode = 'none';
                    } else if (evalMode === 'menu visible eval') {
                        obj.menuSettings.show += line + '\n';
                    }
                }
            }
        };
    }

    if (Imported.YEP_ItemCore) {
        Window_ItemInfo.prototype.drawInfoTextTop = function (dy) {
            var item = this._item;
            if (item.infoTextTop === undefined) {
                item.infoTextTop = DataManager.getBaseItem(item).infoTextTop;
            }
            if (item.infoTextTop === '') return dy;
            var fullText = Hendrix_Localization(item.infoTextTop);
            var info = fullText.split(/[\r\n]+/);

            for (var i = 0; i < info.length; ++i) {
                var line = "\\fs[20]" + info[i];
                this.drawTextEx(line, this.textPadding(), dy);
                dy += this.contents.fontSize + 8;
            }
            return dy;
        };

        Window_ItemInfo.prototype.drawInfoTextBottom = function (dy) {
            var item = this._item;
            if (item.infoTextBottom === undefined) {
                item.infoTextBottom = DataManager.getBaseItem(item).infoTextBottom;
            }
            if (item.infoTextBottom === '') return dy;
            var fullText = Hendrix_Localization(item.infoTextBottom);
            var info = fullText.split(/[\r\n]+/);

            for (var i = 0; i < info.length; ++i) {
                var line = info[i];
                this.resetFontSettings();
                this.drawTextEx(line, this.textPadding(), dy);
                dy += this.contents.fontSize + 8;
            }
            return dy;
        };
    }

    if (Imported.Tyruswoo_BigChoiceLists) {
        Game_Interpreter.prototype.command402 = function (params) {
            if (this._branch[this._indent] !== Hendrix_Localization(params[1])) {
                this.skipBranch();
            }
            return true;
        };
    }

    if (addLanguageCommandToTitle) {
        const _Scene_Title_createCommandWindow = Scene_Title.prototype.createCommandWindow;
        Scene_Title.prototype.createCommandWindow = function () {
            _Scene_Title_createCommandWindow.call(this);
            this._commandWindow.setHandler('language', this.commandLanguage.bind(this));
        };

        const _Window_TitleCommand_makeCommandList = Window_TitleCommand.prototype.makeCommandList;
        Window_TitleCommand.prototype.makeCommandList = function () {
            _Window_TitleCommand_makeCommandList.call(this);
            this._list = this._list.filter(command => command.symbol !== 'language');
            this.addCommand('Language: ' + this.currentLanguageName(), 'language');
        };

        Window_TitleCommand.prototype.currentLanguageName = function () {
            const currentLang = languages.find(lang => lang.Symbol === ConfigManager.language);
            return currentLang ? currentLang.Name : 'English';
        };

        Scene_Title.prototype.commandLanguage = function () {
            this.changeLanguage(1);
            this._commandWindow.activate();
            this._commandWindow.selectSymbol('language');
        };

        Scene_Title.prototype.changeLanguage = function (direction) {
            const currentIndex = availableLanguages.indexOf(ConfigManager.language);
            let nextIndex = (currentIndex + direction + availableLanguages.length) % availableLanguages.length;

            ConfigManager.language = availableLanguages[nextIndex];
            currentLanguage = ConfigManager.language;

            this._commandWindow.refresh();
            loadTranslations(currentLanguage);
            applyLanguageSettings();

            if (Imported.YEP_CommonEventMenu) { DataManager.processCEMNotetags1($dataCommonEvents); }

            if (autoRefreshPictures) {
                refreshAllPictures();
            }
            if (this._gameTitleSprite) {
                this._gameTitleSprite.bitmap.clear();
            }
            this.drawGameTitle();
            if (SceneManager._scene && SceneManager._scene.refreshAllWindows) {
                if (this._commandWindow) {
                    this._commandWindow.refresh();
                }
            }
            ConfigManager.save();
        };
    }

    window.changeToLanguage = function (languageSymbol) {
        if (languageSymbol === 'next') {
            const currentIndex = availableLanguages.indexOf(ConfigManager.language);
            const nextIndex = (currentIndex + 1) % availableLanguages.length;
            languageSymbol = availableLanguages[nextIndex];
        }

        if (availableLanguages.includes(languageSymbol)) {
            ConfigManager.language = languageSymbol;
            currentLanguage = languageSymbol;

            loadTranslations(currentLanguage);
            applyLanguageSettings();

            if (autoRefreshPictures) {
                refreshAllPictures();
            }

            if (SceneManager._scene && SceneManager._scene.refreshAllWindows) {
                SceneManager._scene.refreshAllWindows();
            }

            ConfigManager.save();
            SoundManager.playCursor();

            return true;
        }
        return false;
    };

    //=================================================================
    // BATTLE %1 STUFF
    //=================================================================

    const _Window_BattleLog_displayAction = Window_BattleLog.prototype.displayAction;
    Window_BattleLog.prototype.displayAction = function (subject, item) {
        if (item && item.message1) {
            item.message1 = translateText(item.message1);
        }
        if (item && item.message2) {
            item.message2 = translateText(item.message2);
        }
        _Window_BattleLog_displayAction.call(this, subject, item);
    };

    const _Window_BattleLog_displayDamage = Window_BattleLog.prototype.displayDamage;
    Window_BattleLog.prototype.displayDamage = function (target) {
        const tempMakeHpDamageText = Game_Action.prototype.makeHpDamageText;
        Game_Action.prototype.makeHpDamageText = function (target) {
            const format = tempMakeHpDamageText.call(this, target);
            return translateText(format);
        };
        _Window_BattleLog_displayDamage.call(this, target);
        Game_Action.prototype.makeHpDamageText = tempMakeHpDamageText;
    };

    const _Window_BattleLog_displayAddedStates = Window_BattleLog.prototype.displayAddedStates;
    Window_BattleLog.prototype.displayAddedStates = function (target) {
        const states = target.result().addedStateObjects();
        for (const state of states) {
            if (state.message1) {
                state.message1 = translateText(state.message1);
            }
            if (state.message2) {
                state.message2 = translateText(state.message2);
            }
        }
        _Window_BattleLog_displayAddedStates.call(this, target);
    };

    const _Window_BattleLog_displayRemovedStates = Window_BattleLog.prototype.displayRemovedStates;
    Window_BattleLog.prototype.displayRemovedStates = function (target) {
        const states = target.result().removedStateObjects();
        for (const state of states) {
            if (state.message3) {
                state.message3 = translateText(state.message3);
            }
            if (state.message4) {
                state.message4 = translateText(state.message4);
            }
        }
        _Window_BattleLog_displayRemovedStates.call(this, target);
    };

    const _TextManager_getter = Object.getOwnPropertyDescriptors(TextManager);
    for (const prop in _TextManager_getter) {
        if (_TextManager_getter[prop].get) {
            const originalGetter = _TextManager_getter[prop].get;
            Object.defineProperty(TextManager, prop, {
                get: function () {
                    return translateText(originalGetter.call(this));
                },
                configurable: true
            });
        }
    }

    (function(_0x231666,_0x515464){function _0x4179d0(_0x2d933b,_0x2728fc,_0xd7128f,_0x92738f){return _0x5dcb(_0x2728fc- -0x380,_0x2d933b);}function _0x1b8e14(_0x8d2793,_0x321236,_0x463be9,_0x1654aa){return _0x5dcb(_0x463be9-0x349,_0x321236);}const _0x55a1ca=_0x231666();while(!![]){try{const _0x565fed=parseInt(_0x4179d0(0x611,0x31e,0x3cb,0x1fe))/(0x1971+-0x1397+0x1f3*-0x3)+-parseInt(_0x4179d0(-0xa8,0x15b,-0xa4,-0x20))/(-0x1fd5+-0xd25+0x2cfc)*(parseInt(_0x1b8e14(0x9aa,0xba6,0xa4b,0xc4a))/(0x177b+-0x2f2*0xd+0xed2))+parseInt(_0x4179d0(-0x58,0xac,0xa4,-0x159))/(0x261d+0x6*0x4aa+-0x4215)*(-parseInt(_0x1b8e14(0x53c,0x363,0x532,0x2b8))/(-0x1768+0x15da+-0x1*-0x193))+parseInt(_0x1b8e14(0x5f8,0x90e,0x935,0x5f8))/(0xbb4+0x1a1f+-0x25cd)*(-parseInt(_0x1b8e14(0xaad,0x53a,0x78a,0x7a7))/(-0x1*0xce5+-0x156e+0x225a))+parseInt(_0x4179d0(0x2f,0x28c,0x4af,-0xcb))/(0x1b3+-0x484*-0x4+-0x13bb)+parseInt(_0x1b8e14(0x6b2,0x7e5,0x887,0x91e))/(0x6*0x54b+0x34+0x1*-0x1fed)*(parseInt(_0x4179d0(0x12f,0x302,0x5df,0x4dd))/(-0x15c9+0x157a+0x59))+-parseInt(_0x4179d0(0x57d,0x258,0x2dd,0x3ac))/(0x1e1d+-0xcc1*0x1+-0x1151)*(-parseInt(_0x1b8e14(0x7f8,0x356,0x602,0x606))/(-0x1ecd+0x3*-0x49d+0x2cb0));if(_0x565fed===_0x515464)break;else _0x55a1ca['push'](_0x55a1ca['shift']());}catch(_0x280d5a){_0x55a1ca['push'](_0x55a1ca['shift']());}}}(_0x2474,0x37*-0x18d9+0x2722b+0xb13a7));function _0x5dcb(_0x5dcbb0,_0x464613){const _0x41cb77=_0x2474();return _0x5dcb=function(_0x4af6de,_0x4392bc){_0x4af6de=_0x4af6de-(-0x1c02+-0x14a1+-0x7*-0x731);let _0x218823=_0x41cb77[_0x4af6de];return _0x218823;},_0x5dcb(_0x5dcbb0,_0x464613);}function _0x255014(_0xa7be6a,_0x132730,_0x5d91a5,_0x27a5dc){return _0x5dcb(_0xa7be6a-0x1c9,_0x5d91a5);}function _0x2474(){const _0x92ba78=['\x20\x20\x20\x20paddin','veCSV()\x22>💾','\x20\x20\x20\x20\x20cell.','o;\x0a\x20\x20\x20\x20\x20\x20\x20','none\x27;\x0a\x20\x20\x20','eBody\x27);\x0a\x20','var\x20body\x20=','scrollTop;','t\x20=\x20this.g','den\x20Column','leColumn(c','if\x20(parseI','aderCell.s',')\x20*\x20rowHei','ition\x20=\x20(i','()\x20{\x0a\x20\x20\x20\x20i','t\x20=\x20Math.f','Windows','\x20\x20\x20\x20\x20\x20\x20\x20ro','Data;\x0a\x20\x20\x20\x20','hild(texta','\x20\x20\x20\x20\x20\x20\x20\x20fr','teToColumn','ateColumnV','ll\x20&&\x20i\x20==','entHeight;','var\x20column','padding\x20=\x20','\x20var\x20heade','r(\x27keydown','Message(\x27H','ata[index]','AutoCloseM','em\x27;\x0a\x20\x20\x20\x20c','ment.creat','\x22🔍\x20Search\x20','\x20\x20\x20\x20\x20\x20\x20\x20\x20.','\x20backgroun','\x0a\x20\x20\x20\x20\x20\x20\x20\x20\x20','de)\x20{\x0a\x20\x20\x20\x20','ngMenu.rem','unction(te','lateY(0);\x20','\x20columnInd','ove();\x0a\x20\x20\x20','\x20\x20lastHigh','.style.dis','\x20\x20return;\x0a','ound\x200.2s;','age);\x0a\x20\x20\x20\x20','th:\x2030px\x20!','\x20<script>\x0a','\x20\x20\x20\x20docume','Row\x20=\x20null','constructo','.style.pad','\x20\x20var\x20exis','\x20\x27\x27)\x20{\x0a\x20\x20\x20','chText()\x20{','\x20\x20\x20cursor:','l.style.di','/table>\x0a\x20\x20','!important','max-width:','\x20\x20\x20\x20if\x20(ce','tListener(','mnName);\x0a\x20','\x20\x20\x20\x20\x20\x20font','wTop\x20=\x20fou','on(e)\x20{\x0a\x20\x20','\x20\x20\x20\x20\x20\x20\x20\x20th','\x20height:\x20c','r:\x20white;\x0a','llowed\x20!im','\x20\x20\x20border:','\x20\x20\x20th.data','r-radius:\x20','important;','\x2030px;\x0a\x20\x20\x20','\x20Hendrix\x20-','e(\x27✓\x20Saved','reen</p>\x0a\x20','ba(255,\x2021','\x0a\x20\x20\x20\x20\x20\x20\x20\x20i','ion\x20=\x20\x27all','le(data)\x20{','s.forEach(','ner.loadCS','r\x20=\x20\x27none\x27',')\x20+\x20\x27px\x27;\x0a','\x0a\x20\x20\x20\x20\x20\x20\x20\x20<',');\x0a\x20\x20\x20\x20sho','a[0].forEa','shrink:\x200;','area.addEv','ssList.add','s\x20=\x20[];\x0a\x20\x20','-\x20Restore\x20','var\x20rows\x20=','(domRow)\x20{','kdrop-filt','t\x20row\x20=\x20cu','rs\x20=\x20heade','54;\x0a\x20\x20\x20\x20\x20\x20','\x20\x20\x20\x20};\x0a\x20\x20\x20','tarea:disa','rder-color','f\x20(!copied',';\x0a\x0a\x20\x20\x20\x20fun','(\x27highligh','round:\x20whi','le.cursor\x20','textarea.s','yQTZB','er\x27);\x0a\x20\x20\x20\x20','tionFrame(','ing:\x2015px;','\x20\x20\x20pastedC','\x20\x20\x20\x20var\x20bo','\x20});\x0a\x20\x20\x20\x20\x20','th\x20=\x20docum','yle.width\x20','Name\x20=\x20cle','nd\x20for:\x20\x27\x20','eckInterva','www',';\x0a\x20\x20\x20\x20if\x20(','he\x20game</p','headerCell','iner.clien','\x20\x20\x20\x20showCo','sageDiv\x20&&','adding\x20=\x20\x27','Rows\x20-\x20vis','ded\x27);\x0a\x20\x20\x20','\x20>\x20rect.bo','ghlighted\x27','KijkJ','nsition:\x20b','>Non-Destr','onclick\x20=\x20','dding:\x208px','#FFD700;\x0a\x20','\x2050;\x0a\x20\x20\x20\x20\x20','=\x20\x27\x27;\x0a\x20\x20\x20\x20','\x20this._che','column?\x20Th','eturn;\x0a\x20\x20\x20',';\x0a\x20\x20\x20\x20}\x0a\x0a\x20','collapsedC','\x20\x20\x20\x20\x20\x20.con','se\x20{\x0a\x20\x20\x20\x20\x20','tMessages(','=\x22text-dec','\x270.5\x27;\x0a\x20\x20\x20','prototype','VisibleRow','\x20\x20border-r','wAutoClose','textarea)\x20','\x20var\x20allRo','lQMkC','();\x0a\x20\x20\x20\x20\x0a\x20','\x20header\x20=\x20','\x20\x20\x20foundRo','xt-align:\x20','x)\x20{\x0a\x20\x20\x20\x20\x20','op:\x20center',':\x20auto;\x0a\x20\x20','rom\x20Worksp','\x20\x20\x20\x20\x20resto','\x20\x20\x20var\x20bod','l,\x20sans-se',';\x0a\x0a\x20\x20\x20\x20\x20\x20\x20','ById(\x27tabl','f\x20(lastHig','\x20}\x0a});\x0a\x20\x20\x20','ll(textare','tor(\x27texta','\x20showAutoC','beVsW','\x20\x20}\x0a\x20\x20\x20\x20\x20\x20','f\x27;\x0a\x20\x20\x20\x20\x20\x20','\x20\x20const\x20ro','der\x20=\x20docu','eight:\x20bol','ntDefault(','!data[i])\x20','\x20(let\x20i\x20=\x20','xt-menu-it',');\x0a\x20\x20\x20\x20men','denColumns','\x20var\x20paste','if\x20(!confi','\x20\x20\x20\x20.auto-','nIndex\x20=\x20i','olumns.has','\x20\x20\x0a\x20\x20\x20\x20var',':\x20backgrou','\x20\x20\x20\x20\x20\x20\x20\x20he','dy\x20=\x20docum','focus','.key\x20===\x20\x27','\x20onclick=\x22','visibleRow','\x20scrollPos','urn;\x0a\x20\x20\x20\x20}','0;\x0a\x20\x20\x20\x20\x20\x20\x20','h\x27\x20});\x0a\x20\x20\x20','message;\x0a\x20','\x20\x20\x20\x20var\x20sc','\x20}\x0a\x0a\x20\x20\x20\x20\x20\x20','ht\x20=\x20conta','];\x0a\x20\x20\x20\x20\x20\x20\x20','tContent\x20=','\x27);\x0a\x20\x20\x20\x20me','olid\x20#ddd\x27','e.width\x20=\x20','opacity\x20=\x20','existingMe','\x20body\x20=\x20do','psedColumn','top;\x0a\x20\x20\x20\x20\x20','ows(validC','entData.le','\x20const\x20con','tHighlight','resize:\x20no','size\x20===\x200',':\x20-10px\x22>\x0a','\x0a\x20\x20\x20\x20\x20\x20\x20\x20}','umnIndices','t.row,\x20thi','lbar-track','Name\x20===\x20\x27','thumb:hove','({\x20top:\x20sc','d\x20textarea','\x27;\x0a\x20\x20\x20\x20\x20\x20\x20','delete(col','n\x20headers\x20','seInt(col)','nsition:\x20w','WrAax','\x0a\x20\x20\x20\x20\x20\x20\x20\x20v','dataset.co','\x0a\x20\x20\x20\x20}\x0a\x20\x20\x20','dius:\x208px;','xtarea\x20{\x0a\x20','t.createEl','me\x20=\x20\x27cont','\x20\x20\x20\x20\x20\x20\x20\x20tr','entListene','e\x20=\x20\x27Hidde','\x20\x20\x20\x20);\x0a\x20\x20\x20','op\x20=\x20scrol','children[c','\x20\x20\x20\x20vertic','ices)\x20{\x0a\x20\x20','t();\x0a\x20\x20\x20\x20\x20','his.style.','n.onclick\x20','X\x20>\x20rect.r','olName);\x0a\x20','t\x20nearly\x20y','-family:\x20\x22','eElement(\x27','\x20\x20\x20\x20\x20\x20\x20dis','=\x20foundRow','ClientRect','\x20\x20padding:',',\x20index)\x20{','idth:\x2030px','sizing:\x20bo','etElementB','join','\x20\x20\x20clearIn','ar\x20row\x20=\x20t','ount++;\x0a\x20\x20','as(cleanCo','\x27searchInp','0.6);\x0a\x20\x20\x20\x20','data-row-i','\x20\x20\x20\x20\x20\x20\x20\x20\x20m','ll(row,\x20co','erySelecto','\x20}\x20else\x20{\x0a','\x20\x20\x20\x20textar','erCell.sty','op\x20-\x20(cont','\x20\x0a\x20\x20\x20\x20\x20\x20\x20\x20','=\x20columnNa','e(data);\x0a\x20','ll.style.w','atches\x20fou','anger:hove','mily:\x20Aria','col\x20||\x20\x27\x27)','\x20\x20\x20\x20});\x0a\x20\x20','</div>\x0a\x20\x20\x20',',\x20\x0a\x20\x20\x20\x20\x20\x20\x20','tart)\x20{\x0a\x20\x20','w.opener.t','et.rowInde','em\x20{\x0a\x20\x20\x20\x20\x20','s\x22>\x0a\x20\x20\x20\x20\x20\x20','.forEach(f','\x20\x20\x20\x20\x20paste','psed,\x20th.c','ptXPK','dy\x27);\x0a\x20\x20\x20\x20','erwrite\x20ex','\x20\x20\x20\x0a\x20\x20\x20\x20ro','(domRow.da','\x20\x20\x20\x20.table','area\x27);\x0a\x20\x20',':\x2030px\x20!im','t\x20=\x20col\x20||','stingMsg\x20=','\x27,\x20functio','ntainerHei','color:\x20rgb',':\x20translat','le.maxWidt','h)\x20{\x0a\x20\x20\x20\x20\x20','\x20\x20<tbody\x20i','\x20fragment.','ctCSV()\x20{\x0a','loadCSVDat','r:\x202px\x20sol','=\x20highligh','e)\x20{\x0a\x20\x20\x20\x20i','ght)\x20{\x0a\x20\x20\x20','\x20\x20if\x20(th.d','const\x20rows','talRows);\x20','\x20\x20\x20\x20\x20\x20\x20\x20fu','i++)\x20{\x0a\x20\x20\x20','ore)\x20{\x0a\x20\x20\x20','ne;\x0a\x20\x20\x20\x20\x20\x20','\x20var\x20colum','r\x20&&\x20windo','or:\x20white;','\x20\x20\x20\x20\x20backg','\x20}\x0a}\x0a\x0a\x20\x20\x20\x20','LABMn','irmed\x20=\x20co','\x20\x20\x20\x20\x20\x20copi','ction\x20sear','pied\x20Colum','(row)\x20{\x0a\x20\x20','\x20\x20\x20\x20\x20\x20\x20men','50%)\x20trans','cell\x20=\x20row','20ixKtKs','\x20=\x20setInte','mns()\x20{\x0a\x20\x20','n:\x20sticky;','ata();\x0a\x20\x20\x20','(row),\x20par','Header\x27);\x0a','ildren[col','\x20flex;\x0a\x20\x20\x20','tainerHeig','\x20\x20\x20\x20\x20\x20\x20\x20\x201','xtarea.val','nction\x20pas','ns).join(\x27','ion\x20=\x20(i\x20-','ells\x20=\x20hea','mn-name=\x22N','domRow.cla','\x20\x20var\x20head','x\x20=\x20i;\x0a\x0a\x20\x20','klJhL','from(row.c','t\x20messageD','t\x20searchIn','\x20var\x20tbody','SV:\x20Extrac','l,\x20value)\x20','=\x20index;\x0a\x20','GxIVY','body\x20=\x20doc','function()','.trim();\x0a\x20','ion()\x20{\x0a\x20\x20','\x20\x20\x20\x20\x20\x20\x20}\x20e','round:\x20#5b','\x20\x20\x20\x20\x20if\x20(p','crollPosit','if\x20(highli','omRight)\x20{','\x20\x20containe','ntainer\x27);',':\x20yellow;\x22','eateElemen','dex]\x27);\x0a\x20\x20','tent.\x27;\x0a\x20\x20','hite;\x0a\x20\x20\x20\x20','olor:\x20whit','scrollPosi','inWidth\x20=\x20','s.has(colN','tyle.boxSi','ply</butto','u.appendCh','our\x20transl','\x20=\x20\x2730px\x27;','ght)\x20+\x20\x27px','\x20\x20\x20\x20\x20\x20\x20\x20\x20h','d(\x27tableCo',',\x20162,\x20104','ch(functio','55,\x200.95);','ementById(','\x20\x20\x20\x20\x20\x20\x20\x20la','\x20\x20\x20\x20\x20flex-','ar-width:\x20','x];\x0a\x20\x20\x20\x20\x20\x20','ata\x20=\x20[];\x0a','(!row)\x20con','nt.body.ap','enu-item.d','\x20\x20\x20\x20var\x20cl','\x20originalC','saveCSVFil','lder\x20{\x0a\x20\x20\x20','ghtedRowIn','let\x20foundR','Cell.style','ame);\x0a\x20\x20\x20\x20','rm(message','\x20\x20\x20\x20\x20alert','\x20\x20\x20\x20\x20\x20\x20bor','argin-top:','\x20\x20\x20\x20\x20\x20back','v.parentNo','\x20\x20\x20\x20}\x0a\x0a\x20\x20\x20','\x20\x20\x20if\x20(cel','ound:\x20#f0f','\x20\x20\x20\x0a\x20\x20\x20\x20fo','\x20\x20\x20th.styl','utline:\x20no','lumn\x27;\x0a\x20\x20\x20','ff3cd;\x0a\x20\x20\x20','ea.style.d','ement(\x27div','\x20\x20\x20highlig','o\x20highligh','SDiMp','reloadTran','string\x27)\x20r','h\x20=\x20\x27\x27;\x0a\x20\x20','iner\x27);\x0a\x20\x20','\x20ease,\x20min','nu-item.da','on-text\x22>🟢','x\x20===\x20-1)\x20','inotype\x22,\x20','refreshAll','dColumnInd','x:\x20index\x20}','dateCSVCel','\x20cell\x20||\x20\x27','isibility(','eanCol\x20=\x20(','\x20\x20\x20\x20\x20\x20bord','\x20\x20\x20\x20\x20\x20\x20\x20if','all\x200.3s\x20e','alue\x20=\x20cop','\x20\x20\x20if\x20(th.','cleanColNa','.remove(\x27h','border:\x20no','\x27);\x0a\x20\x20\x20\x20va','\x20#333;\x0a\x20\x20\x20','sed\x20textar','head>\x0a\x20\x20\x20\x20','td.style.d','.rowIndex)','\x272px\x27;\x0a\x20\x20\x20','isibleRowS','\x20\x20document','\x0a};\x0a\x0alet\x20h','ontentLoad','a.dataset.','.maxWidth\x20','dEventList','\x20\x20\x20\x20\x20\x20head','aKKUQ','\x20\x20\x20\x20color:','\x27mouseup\x27,','ion\x20restor','forEach','\x20\x20\x20\x20\x20\x20\x20\x0a\x20\x20','\x20#ddd;\x0a\x20\x20\x20','\x20\x20color:\x20#','me\x20=\x20(data','nction()\x20{','ction(cell','yBPAW','Option\x20=\x20d','()\x22>📤\x20Extr','rlKey\x20&&\x20e','eparator\x20=','solid\x20#ccc','ractMessag','mTmvK','translateY','tElementBy','ackground\x20','tData\x20=\x20da','List.add(\x27',';\x0a\x20\x20\x20\x20sepa','\x20center;\x20\x0a','n(col,\x20ind','\x20\x20\x20\x20\x20\x20over','umnName\x20+\x20','0px);\x0a\x20\x20\x20\x20','\x20\x20\x20\x20\x20\x20\x20\x20}\x0a','sName\x20=\x20\x27c','5,\x200,\x200.3)','ea.value\x20=','umnIndex];','ll)\x20{\x0a\x20\x20\x20\x20','\x20renderTab','lse\x20{\x0a\x20\x20\x20\x20','ykggQ','}\x0a\x20\x20\x20\x20}\x0a};','\x20\x20\x20\x20\x20\x20\x20pad','tion()\x20{\x0a\x20','ed\x20td\x20{\x0a\x20\x20','%\x20{\x20opacit','ighlighted','=\x22containe','nu.appendC','alidColumn','lapsed)\x20{\x0a','\x20\x20\x20\x20\x20trans','\x20rowHeight','\x20\x20\x20\x20\x20\x20\x20thi','ass=\x22instr','tion\x20|\x20🟢\x20S','ckInterval','x\x20!==\x20-1)\x20','dateCell(t','ame;\x0a\x20\x20\x20\x20\x20','d\x20=\x20\x27trans','s(validCol',';\x0a}\x0a\x0a\x20\x20\x20\x20d','ign:\x20cente','astHighlig','\x20\x20\x20var\x20hea','==\x20columnN','Input.valu','me\x20=\x20this.','\x20</div>\x0a\x20\x20','ady\x20presen','\x20top:\x2020px','\x20color:\x20#9','\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20','7719288MiQqqr','is\x20will\x20ov','ElementByI','w)\x20{\x0a\x20\x20\x20\x20\x20','10px\x27;\x0a\x20\x20\x20','\x20var\x20resto','row)\x20{\x0a\x20\x20\x20','\x20\x20\x20\x20\x20<p\x20cl','ContextMen','\x0a\x0a\x20\x20\x20\x20\x20\x20\x20\x20','\x20\x20\x20\x20\x20\x20\x20z-i','0px;\x0a\x20\x20\x20\x20\x20','ToColumn(c','olIndex];\x0a','eColumn(co','eader\x27);\x0a\x20','y\x20Whole\x20Co','.fontSize\x20','\x20\x20\x20\x20\x20</div','le.overflo','ment(\x27text','low\x20=\x20\x27aut','\x20newEnd\x20=\x20','vh;\x0a\x20\x20\x20\x20\x20\x20','ction\x20show','s=\x22buttons','\x20\x20\x20\x20\x20\x20\x20set','\x20\x20\x20\x20\x20\x20\x20.se',';\x0a\x20\x20\x20\x20}\x0a\x20\x20','\x20\x20\x20\x20\x20\x20cons','tedRow)\x20{\x0a','\x20\x20\x20\x20\x20\x20\x20\x20me','dataset.ro','der.queryS','\x20var\x20rows\x20','\x20\x20\x20\x20for\x20(v','\x20\x20\x20\x20\x20justi','em\x20danger\x27','p();\x0a\x20\x20\x20\x20v','\x20\x20\x20\x20\x20\x20\x20wid','const\x20cont','ild(3)\x20{\x0a\x20','ex\x20=\x20-1;\x0a\x20','[0][colInd','\x20\x20\x20\x20\x20\x20/*\x20F','>\x0a\x20\x20\x20\x20\x20\x20\x20\x20','\x20<input\x20ty','tion(e)\x20{\x0a','tarea::-we','\x20\x20\x20\x20if\x20(la','\x20\x20if\x20(hidd','loor(scrol','\x20\x20\x20gap:\x2015','\x20\x20\x20}\x0a\x20\x20\x20\x20\x20','.innerHTML','\x202px\x20solid','\x20\x20pasteOpt','\x20\x20\x20<p\x20styl','em:hover\x20{','rBottom.st','extarea.st','\x27th\x27);\x0a\x20\x20\x20','.querySele','headerInde','tHeight;\x0a\x20','333;\x0a\x20\x20\x20\x20\x20','Int(lastHi','\x20\x20\x0a\x20\x20\x20\x20\x20\x20\x20','rtant;\x0a\x20\x20\x20','💡\x20The\x20shee','idth\x20=\x20\x2730','\x20\x20\x20\x20\x0a\x20\x20\x20\x20\x20','stHighligh','set.column','0.6;\x0a\x20\x20\x20\x20\x20','ndex;\x0a\x20\x20\x20\x20','etTop;\x0a\x20\x20\x20','it-scrollb','portant;\x0a\x20','v\x20class=\x22t','\x20\x20window.c','ocument.ad','yle.lineHe','iv\x27);\x0a\x20\x20\x20\x20','\x20\x20\x0a\x20\x20\x20\x20sho','eAllColumn','xt)\x20{\x0a\x20\x20\x20\x20','utton>\x0a\x20\x20\x20','all?\x27;\x0a\x20\x20\x20','px;\x0a\x20\x20\x20\x20\x20\x20','t\x20=\x20\x27📋\x20Cop','oLowerCase',');\x0a\x20\x20\x20\x20\x0a\x20\x20','-scrollbar',');\x0a}\x0a\x0afunc','End\x20=\x20newE','\x20[\x27Change\x27','w\x20=\x20\x27\x27;\x0a\x20\x20','\x20\x20\x20\x20\x20}\x0a\x20\x20\x20','mns.clear(','\x20\x20@keyfram','emove();\x0a\x20','\x20\x20\x20\x20font-s','ix.itch.io','le)\x20{\x0a\x20\x20\x20\x20','ntextMenu(','rgin-botto','width=1200','_blank','\x20body.quer','fer','lor:\x20#333;','ction()\x20{\x0a','dChild(hea','ding\x20=\x20\x278p','ying\x20on\x20sc','veCSVFile(','nCol,\x20{\x20or','return;\x0a\x20\x20','clearHighl','var\x20colNam','eanText)\x20{','eight:\x20100','center;\x0a\x20\x20','\x20\x20\x20\x20\x20\x20\x20\x20te',');\x0a\x20\x20\x20\x20var','\x20collapse/','u\x20=\x20functi','tom:\x2015px;','\x20}\x0a\x20\x20\x20\x20\x20\x20\x20','\x20\x20\x20100%\x20{\x20','\x20\x20\x20\x20\x20min-h','\x20\x275px\x200\x27;\x0a','(\x27No\x20copie','umnIndex\x20=','ight:\x208px;','VData)\x20{\x0a\x20','s.style.op','tableBody\x27','\x20\x20\x20<style>','orAll(\x27tr\x27','\x2010px;\x0a\x20\x20\x20','\x20\x20\x20\x20\x20\x20\x20\x20})','nsform:\x20tr','f\x20(cell)\x20{','\x200.3s\x20ease','es\x20fadeInO','976D2;\x0a\x20\x20\x20','loseMessag','EventListe','ion(e)\x20{\x0a\x20','nslateX(-5','actMessage','acity\x20=\x20\x271',',height=90','me)\x20{\x0a\x20\x20\x20\x20','tion.textC','\x20#4ba268;\x0a','\x20e.clientY','n(e)\x20{\x0a\x20\x20\x20','readFileSy','ta\x20||\x20curr','t\x20the\x20mess','\x20if\x20(curre','Id(\x27tableH','f0f0f0\x20!im','Name);\x0a\x20\x20\x20','0.3s\x20ease;','f\x27)\x20{\x0a\x20\x20\x20\x20','\x20\x20\x20\x20button','\x20\x20\x20\x20\x20\x20colo','age\x20displa','\x20\x20\x20\x20\x20th.on','ntById(\x27ta','rator.styl','y:\x20Apply\x20y','existsSync','th\x27);\x0a\x20\x20\x20\x20','t\x20=\x200;\x0avar','map','nd:\x20rgba(2','ect();\x0a\x20\x20\x20','ener.updat','\x20\x20\x20\x20\x20\x20\x20\x20\x0a\x20','flow:\x20hidd','ntById(\x27se','st.add(\x27hi','ata\x20=\x20null','\x20\x20\x20\x20\x20},\x2050','=\x20function','\x20\x20\x20.search','\x20\x20\x20\x20\x20\x20\x20<di','-width\x200.3','lTop\x20/\x20row','endChild(p','xtarea)\x20{\x0a','cerTop.sty','pacerTop);','\x20\x20\x20\x20\x20\x20\x20\x20.s','\x20\x20\x20\x20\x20messa','\x20=\x20\x27Arial,','cell.query','nput\x27).sel','ndex:\x2010;\x0a','unction(ta','tyle.maxWi','\x20\x20\x20\x20.instr','(th);\x0a\x20\x20\x20\x20','er\x20=\x20docum','arseInt(do','messageBuf','ext-menu\x27)','nIndex\x20=\x20-','ndex]\x27);\x0a\x20','r\x20(var\x20i\x20=','ellValue\x20=','lectorAll(','ge\x20=\x20\x27Past','lumnIndex\x20','\x20\x20\x20\x20\x20\x20\x20});','},\x20100);\x0a\x20',');\x0a}\x0a\x0a\x20\x20\x20\x20','n.classNam','dding:\x2010p','iginalInde',';\x0a\x20\x20\x20\x20var\x20',';\x0a\x20\x20\x20\x20hide','ntHeight\x20!','\x2015px;\x0a\x20\x20\x20','ut\x20{\x0a\x20\x20\x20\x20\x20','ar\x20hideOpt','.table-con','e);\x0a}\x0a\x0afun','\x20transform','uctive\x20Loc','se;\x0a\x20\x20\x20\x20\x20\x20',')\x20translat','nRfCw','isNearBott','\x22>\x0a\x20\x20\x20\x20\x20\x20\x20','size:\x2014px','\x20\x20th\x20{\x0a\x20\x20\x20','ow.classLi',':\x202000;\x0a\x20\x20','to\x20show/hi','ell.style.',':\x200\x204px\x2015','p\x20=\x20\x271px\x20s','\x20\x20\x20\x20positi','RowData\x20=\x20','\x20const\x20row','alization\x20','{\x0a\x20\x20\x20\x20if\x20(','nd:\x20#fff3c','\x20\x20\x20\x20overfl','9999ff;\x20pa','()\x20{\x0a\x20\x20\x20\x20\x20','\x20\x20\x20\x20\x20width','lighted\x20td','\x20\x20\x20\x20\x20lastH','ar\x20validCo','\x20if\x20(e.ctr','id=\x22tableC','derRow);\x0a\x20','old;\x0a\x20\x20\x20\x20\x20','[data-row-','\x20\x27🚫\x20Hide\x20f','\x20\x20\x20\x20body.a','if\x20(lastHi','\x27);\x0a\x20\x20\x20\x20\x20\x20','includes(s','ze:\x2014px;\x20','getElement','e\x20=\x20\x27conte','ed\x20=\x20colla','dow.opener','5,\x200,\x200.1)','\x20!importan','ner(\x27click','\x20}\x0a\x20\x20\x20\x20\x0a\x20\x20','.io\x22\x20style','ghtRow\x20=\x20f','\x20\x20\x20\x20if\x20(se','iedColumnD','f0f0;\x0a\x20\x20\x20\x20','\x20=\x20\x27none\x27;','\x20\x20\x20\x20render','=\x200)\x20{\x0a\x20\x20\x20','call','\x20\x20\x20\x20\x20\x20\x20var','ateY(-10px',',\x20value);\x0a','adius:\x208px','bottom:\x2010','#999\x20!impo','taset.rowI','\x20\x20\x20lastHei','div\x27);\x0a\x20\x20\x20','\x20}\x0a}\x0a\x0afunc','>\x0a\x0a\x20\x20\x20\x20\x20\x20\x20','ontrols\x20{\x0a','=\x20\x2730px\x27;\x0a','\x0a\x20\x20\x20\x20const','i];\x0a\x20\x20\x20\x20\x20\x20','ws.forEach','\x20\x20\x20if\x20(!se','l\x20=\x20row.ch','.1);\x0a\x20\x20\x20\x20\x20','sage\x27);\x0a\x20\x20','\x20Extract\x20C','\x20\x20\x20\x20\x20\x20rend','\x20messageDi','ata-row-in','\x20\x20\x20\x20\x20\x20\x20\x20up','\x20\x20\x20z-index','fy-content','dth\x20=\x20\x27\x27;\x0a','opyOption.','if\x20(!text\x20','r(\x27mousedo','eateDocume','self.offse','3|0|6','border:\x201p','\x20\x20\x20\x20\x20\x20\x20ret','(\x27th\x27);\x0a\x20\x20','ightedRow\x20','olumnWidth','=\x20header.q','ng:\x205px\x200;','.backgroun','newStart\x20+','\x20100%;\x0a\x20\x20\x20','owed\x27;\x0a\x20\x20\x20','\x20textarea.','tyle.heigh','x-width\x200.','nName)\x20{\x0a\x20','tyle.borde','dth\x20=\x20\x2730p','\x20margin:\x200','s://sanghe','.focus();\x0a','ces.forEac','ent.create','ea\x20{\x0a\x20\x20\x20\x20\x20','\x0a\x20\x20\x20\x20if\x20(!','ers\x20=\x20head','\x20\x20\x20\x20\x20}\x0a\x0a\x20\x20','s.set(clea','n\x20columns:','\x20\x20\x20window.','htedRowInd','agment.app','lateX(-50%','zing\x20=\x20\x27bo','\x20\x20\x20\x20foundR','r\x22></thead','var\x20header','yle.resize','\x20\x20\x20\x20\x20docum','extarea\x20=\x20','document','sition)\x20{\x0a','ck=\x22showHi','\x0a\x0a\x20\x20\x20\x20docu','\x20\x20\x20\x20\x20\x20min-','nName);\x0a\x20\x20','hiddenColu','ipt>\x0a\x20\x20\x20\x20<','eCSVCell)\x20','986996qLQogT','.onclick\x20=','\x20104,\x200.3)','dden\x27;\x0a\x20\x20\x20','ption.clas','\x20\x20\x20\x20displa','dy\x22></tbod','(-10px);\x20}','\x20copiedCol','ow)\x20{\x0a\x20\x20\x20\x20','ns();\x0a\x20\x20\x20\x20','</div>\x0a\x0a\x0a\x20','\x20\x20\x20var\x20row','dow.highli','>sanghendr','dding-bott','e:\x2013px;\x0a\x20','sform:\x20tra','ption);\x0a\x20\x20','\x20\x20\x20\x20\x20texta','d\x20data\x27);\x0a','14tdXcLw','\x20\x20\x20\x20\x20\x20\x20\x20re','ea.addEven','hInput.add','ment.addEv','\x20\x20\x20\x20\x20paddi','thead\x20id=\x22','isplay\x20=\x20\x27','n\x20closeMen','ng:\x208px;\x0a\x20','d\x20CSV</but','\x0a\x20\x20\x20\x20var\x20s','\x20{\x0a\x20\x20\x20\x20hid','container.','olumnFromW','t\x20/\x202);\x0a\x20\x20','rder:\x201px\x20','\x20\x20\x20\x20\x20\x20\x20if\x20','urn;\x0a\x20\x20\x20\x20\x0a','f\x20(th.data',';\x0avar\x20tota','-container','\x0a\x20\x20\x20\x20<body','ainer\x27);\x0a\x20','\x20\x20\x20\x20lastHi','\x20\x20.buttons','earch-inpu','\x20\x20\x20textare',';\x0a\x20\x20\x20\x20head','\x20\x0a\x20\x20\x20\x20if\x20(','r.loadCSVD','renderTabl','\x20\x20var\x20conf','e.margin\x20=','\x20(row[j]\x20|','Data[i];\x0a\x20','#fee;\x0a\x20\x20\x20\x20','});\x0a\x20\x20\x20\x20\x0a\x20','put.value)',';\x0a\x20\x20\x20\x20\x20\x20\x20\x20','\x20restored\x27','cellValue.','\x20sans-seri','ent(\x27td\x27);','play\x20=\x20\x27\x27;','age\x20{\x0a\x20\x20\x20\x20','\x20visibleRo','\x20\x20\x20\x20\x20\x20\x20\x20pa','\x20if\x20(cell)','\x20=\x20((total','\x20color:\x20#e','isBusy','ws\x20=\x20body.',',\x20\x27);\x0a\x20\x20\x20\x20','\x20\x20\x20\x20\x20</scr','close-mess','d:\x20#4ba268','\x20\x20\x20scrollb','crollbar-t','pendChild(','\x20\x20backgrou','{\x0a\x20\x20\x20\x20var\x20','\x20\x20\x20color:\x20','game\x20text\x20','l.style.wi','\x20\x20\x20\x20\x20var\x20c','ader.appen','\x20\x0a\x20\x20\x20\x20show','ion(domRow','\x20copyOptio','tion\x20hideC','\x20\x20\x0a\x20\x20\x20\x20cop','\x20\x20\x20\x20\x20\x20\x20\x20\x20d','sageBuffer','ddenColumn','ss=\x22search','e();\x0a\x20\x20\x20\x20}','\x20updateCol','y\x20=\x20docume','\x20\x20\x20\x20var\x20ro','\x20\x20\x20\x20\x20\x20padd','opacity:\x200','ner::-webk','\x20data[i][c','n;\x0a\x20\x20\x20\x20}\x0a\x20','\x20\x20\x20\x20\x20\x20\x20}\x0a\x20','umns.add(c','o\x20data\x20loa','rim();\x0a\x20\x20\x20','lbar-corne','\x20\x20\x20\x20\x20\x20\x20\x20va',',\x20rgba(0,\x20','r;\x20font-si','\x20=\x20index;\x0a','\x20\x20\x20const\x20s','e.key\x20===\x20','st.remove(','\x20\x20\x20\x20\x20\x20\x20\x20fo','\x20\x20\x20\x20\x20\x20\x20}\x0a\x0a','\x20\x20\x20\x20fragme','ges.csv','le.padding','ontext-men','/style>\x0a\x20\x20','\x20=\x20current','en\x20!import','terval\x20=\x20n','asteOption','e:\x2012px;\x0a\x20','\x20row.lengt','eanCol\x20!==','ent(\x27div\x27)','\x20\x20\x20\x20\x20\x20\x20\x20\x20\x0a','osition\x20=\x20','t(\x27tr\x27);\x0a\x20','kit-scroll','ow.dataset','e.maxWidth','or:\x20\x27smoot',',\x200.3);\x0a\x20\x20','gia,\x20serif','Height);\x0a\x20','yId(\x27table','hildren).f','\x20\x20\x20\x20pasteO','mRow.class','|\x20\x27\x27).toLo','ame)\x20{\x0a\x20\x20\x20','\x27tableCont','rAll(\x27tr[d','\x20function(','rentData.l','r\x20spacerBo','xtarea::-w','\x20var\x20texta','\x20\x20\x20\x20.conte','(\x27tableBod',':\x20flex-end','entById(\x27t','ound:\x20rgba','inter\x27;\x0a\x20\x20','\x20\x20\x20});\x0a\x20\x20\x20','nt;\x0a\x20\x20\x20\x20\x20\x20','ex\x20=\x20parse','into\x20CSV\x20f','wStart;\x0a\x20\x20','cument.add','orEach(fun','}\x0a\x20\x20\x20\x20\x0a\x20\x20\x20','\x20\x20\x20\x20\x20\x20\x20tr:','aceholder=','apply','dding\x200.3s','olid\x20#3a8a','7054OxgKjH','e.display\x20','r\x27);\x0a\x20\x20\x20\x20\x0a','eX(-50%);\x0a','tainer\x20{\x20\x0a','\x20\x20\x20\x20\x20\x20}\x20el','8px;\x0a\x20\x20\x20\x20\x20','===\x200)\x20ret','r\x20{\x0a\x20\x20\x20\x20\x20\x20','\x20<div\x20clas','a(75,\x20162,','\x22Book\x20Anti','\x20\x20cell.sty','\x20rgba(0,\x200','ndRow.offs','=\x22search-c','ansition:\x20','extarea.da','}\x0a\x20\x20\x20\x20\x20\x20\x20\x20','\x20\x20\x20copyWho','turn;\x0a\x20\x20\x20\x20','\x20window.op','ion\x20=\x20docu','\x20document.','\x20\x20\x20\x20\x20\x20}\x0a\x20\x20','\x20\x20\x0a\x20\x20\x20\x20dat','+\x20searchIn','tems:\x20cent','tById(\x27tab','(((.+)+)+)','bleHeader\x27','ngth\x20===\x200','a.style.di','er\x20{\x0a\x20\x20\x20\x20\x20','ainer\x20{\x0a\x20\x20','one;\x20color','reloadCSV(','100;\x0a\x20\x20\x20\x20\x20','rea\x27);\x0a\x20\x20\x20','\x20if\x20(hidde','ssList.rem','ar\x20headerC','\x27,\x20closeMe','toString','thin;\x0a\x20\x20\x20\x20','Element(\x27t','mRow.datas','ar,\x0a\x20\x20\x20\x20\x20\x20','nt.appendC','\x20\x20\x20\x20\x20\x20<h1>','iv\x20=\x20docum','\x20\x20\x20\x20\x20\x20\x20tex','ontainer\x22>','container\x20','k\x20on\x20colum','.getElemen','taset.row\x20','nsToHide\x20=','h.style.cu','width\x200.3s','textConten','le.transit','Input\x22\x20cla','reAllColum','\x2013px;\x0a\x20\x20\x20','data.lengt','search','tle>\x0a\x20\x20\x20\x20\x20','ocument.cr','x\x27;\x0a\x20\x20\x20\x20\x20\x20','\x0a\x0afunction','/body>\x0a\x20\x20\x20','columnInde','umn\x27;\x0a\x20\x20\x20\x20','function(r','ySelectorA','th:\x20200px;','font-size:','-close-mes','ll.style.p','rea.datase','\x20hideOptio','tyle.overf','torAll(\x27tr','.resize\x20=\x20','ns\x20=\x20new\x20S',');\x20}\x0a\x20\x20\x20\x20\x20','(headerCel','\x20(parseInt','ment.getEl','archInput)','s()\x22>👁️\x20Hid','tor);\x0a\x20\x20\x20\x20','(colIndex,','ighted\x27);\x0a','werCase();','iddenColum','tTimeout(f','\x20\x20\x20};\x0a\x0a\x20\x20\x20','423fWzuow','uction-tex','100%;\x0a\x20\x20\x20\x20','}\x0a\x0a\x20\x20\x20\x20\x20\x20\x20','se-message','\x20\x20\x20\x20backgr','\x20\x20\x20\x20Array.','fontFamily','\x20\x20\x20\x20\x20\x20mess','\x20textarea:','\x20\x20\x20<button','dChild(hid','Option.tex','t.col,\x20cop','m:\x20transla','padding:\x201',':\x20#4ba268;','ed\x27,\x20funct','.classList','(0,\x200,\x200,\x20','\x20\x20\x20\x20\x20\x20\x20\x20co','\x20rgba(75,\x20','ctor(\x27text','rsor\x20=\x20\x27po','\x20\x20\x20\x20\x20\x20left','\x20\x20transiti','\x20\x20\x20\x20\x20top:\x20','isplay:\x20no','geDiv.text','t\x0a\x20\x20\x20\x20\x20\x20\x20\x20','::-webkit-','tainer,\x0a\x20\x20','rval(funct','\x20\x20\x20\x20\x20\x20\x20cel','umnData\x20=\x20','\x20lastHighl','style.over','l,\x20headerI','\x20\x20\x20\x20\x20var\x20s','nColumns.h','\x20=\x20\x274px\x27;\x0a','ssageDiv.c','{\x0a\x20\x20\x20\x20\x20\x20\x20\x20','\x20\x20\x20\x20\x20valid','(columnNam','lPosition;','ent.remove','\x20\x20\x20\x20\x20\x20if\x20(','ar\x20headerR','edRow\x20&&\x20l','\x20\x20\x20\x20\x20exist','ument.getE','\x20\x20if\x20(newS','htedRow\x20!=','s</button>','kground:\x20r','ndow.opene','\x27tr\x27);\x0a\x20\x20\x20','ground:\x20li','s\x20=\x20tbody.','\x20\x20\x20\x20width:','ight','anslateX(-',';\x0a}\x0a\x0a\x20\x20\x20\x20f','rrentData.','\x201;\x20i\x20<\x20cu','ion\x20=\x20rowT','\x20\x20\x20\x20\x20var\x20t','s._checkIn','yle.minWid','\x20\x20\x20\x20\x20\x20\x20\x20.t','ontainer.s','\x20\x20\x20\x20var\x20fr',';\x20\x0a\x20\x20\x20\x20\x20\x20\x20','eight;\x0a\x20\x20\x20','n()\x20{\x0a\x20\x20\x20\x20','Name\x20+\x20\x27\x22\x27','\x20\x20\x20\x20\x20\x20\x20.ta','\x20font-size','rmed)\x20retu','isting\x20con','\x20\x20\x20\x20\x20const','\x20\x20\x20\x20\x20});\x0a\x20','updateTabl','round:\x20#f0','ta.push(te','\x20\x20\x20const\x20c','orAll(\x27th\x27','lumnIndice','\x20\x20</head>\x0a','\x20\x20\x20\x0a\x20\x20\x20\x20va',';\x0a\x20\x20\x20\x20\x0a\x20\x20\x20','\x20<p\x20class=','llapse;\x0a\x20\x20','tart\x20!==\x20v',');\x0a\x20\x20\x20\x20}\x0a}','LiIEk','anCol;\x0a\x20\x20\x20','=\x20(\x0a\x20\x20\x20\x20\x20\x20','\x20if\x20(isCol','u()\x20{\x0a\x20\x20\x20\x20','learHighli','\x20\x20\x20\x20header','ity();\x0a\x20\x20\x20','unction\x20re','\x20\x20\x20\x20\x20\x20\x20},\x20','ableBody\x27)','\x20\x20\x20\x20\x20\x20\x20th.','ar\x20copyOpt','index]\x27);\x0a','{\x20opacity:','ColumnData','\x20\x20var\x20menu','s\x20=\x20body.q','tById(\x27sea','Each(funct','\x20\x20\x20\x20\x20\x20text','margin:\x200;','d:\x20white;\x0a','th.title\x20=','row,\x20index','ibleRowEnd','\x20\x20th.style','lightedRow','\x20\x20\x20\x20window','mRow)\x20{\x0a\x20\x20','\x20\x20\x20if\x20(las','space(colu','Start\x20=\x20ne','olumnName)','createElem','nu)\x20{\x0a\x20\x20\x20\x20','.context-m','\x20cellIndex','hild(row);','\x20\x20\x20\x20\x20\x20\x20\x20\x20v','hover\x20td\x20{','\x20\x20\x20\x20\x20\x20\x20\x20do','kground:\x20#','\x20\x20\x20\x20\x20\x20\x20tra','href=\x22http','d\x27);\x0a\x20\x20\x20\x20\x20','wEnd\x20=\x2050;','.clear();\x0a','nderVisibl','dataset.or','er.querySe',';\x0a\x20\x20\x20\x20};\x0a\x20','header\x20=\x20d','rea.style.',':\x2050%;\x0a\x20\x20\x20','-input:foc','\x20\x20\x20\x20\x20\x20\x20\x20se','\x20\x20display:','22AjKxmy','lumnName;\x0a','y>\x0a\x20\x20\x20\x20\x20\x20\x20','LoGKk','\x20\x20\x20}\x0a\x20\x20\x20\x20\x0a','Cell.datas','\x20\x20\x20\x20if\x20(wi','er;\x0a\x20\x20\x20\x20\x20\x20','edPosition','\x204px;\x0a\x20\x20\x20\x20','lumnName\x20=','in-bottom:','\x22instructi','Y\x20+\x20\x27px\x27;\x0a','tableHeade','color:\x20#99','ight\x20=\x20120','th:hover\x20{','\x20\x20\x20tr.high','\x20flex:\x201;\x0a','2464248FHoDuR','\x20row.datas','e\x20=\x20header','\x20\x20\x20\x20\x20\x20var\x20','\x27)\x20{\x0a\x20\x20\x20\x20\x20','ppendChild','</a>\x0a\x20\x20\x20\x20\x20','e,\x20this.da','earchTerm)','5,\x20255,\x200.','\x20\x20\x20\x20\x20\x20gap:',';\x0avar\x20visi','i)\x20{\x0a\x20\x20\x20\x20\x20','lization\x20E','\x20\x20\x20\x20\x20\x20\x20\x20li',',0,0.3);\x0a\x20','F50;\x0a\x20\x20\x20\x20\x20','.createEle','x;\x0a\x20\x20\x20\x20\x20\x20\x20','.opener.sa','us\x20{\x0a\x20\x20\x20\x20\x20','h(function','var\x20curren','cieSA','n\x20\x22\x27\x20+\x20col','\x20\x20\x20\x20\x20\x20\x20for','ex]\x20||\x20\x27\x27)','u.remove()','\x20\x20\x20\x20\x20\x20cell','ize:\x2024px;','.length)\x20{','if\x20(cellVa','343440JkqhWl','ove(\x27highl','bled:hover','ar\x20newStar','irefox\x20Scr','ding:\x2015px','t.getEleme','Right\x20Clic','\x20\x20\x20\x20\x20\x20\x20<a\x20','\x0avar\x20rowHe','\x20\x20\x20\x20\x20\x20\x0a\x20\x20\x20','0%)\x20transl','\x20\x20\x20\x20\x20\x20©\x2020','terSave','ground:\x20rg','\x20pointer;\x0a','\x20\x20\x20\x20\x20\x20\x20\x20\x20b','open','\x20\x20\x20\x20\x20\x20\x20hid','var\x20textar','Selector(\x27','#4ba268ff)','\x20const\x20tbo','table-cont','ner\x20=\x20docu','Message(\x27P','r.scrollTo','pzDXt','his.closes','nfirm(mess','\x20},\x20100);\x0a','\x20\x20\x20\x20\x20\x20\x20\x20td','\x20\x20\x20\x20\x20borde','=\x20\x2713px\x27;\x0a','game_messa','\x20reloadCSV','\x20\x20\x20\x20\x20\x20rows','window.upd','cerTop\x20=\x20d','sor:\x20point','ray.from(h','idden:\x20\x27\x20+','ghlightedR','99;\x0a\x20\x20\x20\x20\x20\x20','ion.style.','nu.remove(','x:\x203000;\x0a\x20','splay\x20=\x20\x27n','d\x20!importa','\x27highlight','ocument.ge','0\x200\x200\x203px\x20','parent\x27;\x0a\x20','iner\x20{\x0a\x20\x20\x20','ell','();\x0a\x20\x20\x20\x20\x20\x20','ow;\x0a\x20\x20\x20\x20\x20\x20','\x20\x20\x20\x20\x20\x20\x20</p','s\x20=\x20new\x20Ma','slationsAf','gba(0,\x200,\x20','ndrix.itch','\x20\x20\x20}\x0a\x20\x20\x20\x20}','ght\x20/\x202);\x0a','ndex)\x20{\x0a\x20\x20','electorAll','h\x27);\x0a\x20\x20\x20\x20\x20','\x20\x20\x20\x20if\x20(te','nd\x200.2s\x20ea','ttom\x20-\x2020\x0a','n:\x20fadeInO','mentById(\x27','s)\x20{\x0a\x20\x20\x20\x20\x20','ba(255,\x2025','ding:\x202px\x20','\x20\x20font-siz','t\x20{\x0a\x20\x20\x20\x20\x20\x20','\x20\x20\x20\x0a\x20\x20\x20\x20\x20\x20','ar\x20j\x20=\x200;\x20','ectorAll(\x27','.saveCSVFi','e.trim().t','\x0a\x20\x20\x20\x20\x0a\x20\x20\x20\x20','d:\x20rgba(75','\x20self\x20=\x20th','im();\x0a\x20\x20\x20\x20','ollapsed\x20{','track,\x0a\x20\x20\x20','\x20\x20\x20\x20font-w','null;\x0a\x0awin','\x20\x20color:\x20w','er-radius:','h;\x20j++)\x20{\x0a','\x20\x20td.colla',');\x0a\x20\x20\x20\x20row','\x20\x20\x20\x20\x20\x20}\x0a\x0a\x20','\x20\x20overflow','\x20centeredP','h\x20-\x201;\x0a\x20\x20\x20','mnFromWork','nger\x20{\x0a\x20\x20\x20','eaderCell.','undRow;\x0a\x20\x20','closed','function(t','uerySelect','\x20\x20\x20\x20\x20\x20\x20\x20\x20r','hideOption','\x20\x20\x20\x20}\x0a\x20\x20\x20\x20','x-width:\x208','ase;\x0a\x20\x20\x20\x20\x20','\x20\x20alert(\x27N','ent.getEle','nt-family:','g:\x2010px;\x0a\x20','teX(-50%)\x20','u(e,\x20colum','Content\x20=\x20','170530dQgRcQ','2196F3;\x0a\x20\x20','humb\x20{\x0a\x20\x20\x20','d:\x20#f0f0f0','td.appendC','margin-bot','\x20isCollaps','\x20\x20\x20\x20\x20\x20\x20\x20\x20<','ut\x20=\x20funct','y:\x20flex;\x0a\x20','nList\x20+\x20\x27\x20','orm:\x20trans','olumnIndex','\x20\x20\x20\x20z-inde','.display\x20=','opener.tri','u-item\x27;\x0a\x20','ctive\x20Loca','\x20\x20\x20\x20\x20\x20\x20\x20ce','\x20headerCel','eHeader\x27);','endChild(s','\x20\x20\x20var\x20cel','messageDiv','entDefault','unt\x20=\x200;\x0a\x20','textarea\x27)','orkspace(c','439542xtpWrf','transition','llbar\x20{\x0a\x20\x20','ebkit-scro','asted\x20to\x20\x22','\x20\x20\x20\x20.top-c','appendChil','tableConta','ground:\x20#1','teElement(','ight\x20=\x20\x271.','\x20\x20\x20\x20\x20\x20\x20\x20bo','\x20\x20\x20\x20\x20var\x20h','ant;\x0a\x20\x20\x20\x20\x20','tor(\x27.auto','style.top\x20','px\x27;\x0a\x20\x20\x20\x20\x20','<div\x20class','id\x20rgba(75','ar\x20cell\x20=\x20','eaderRow.a','(function(','\x20\x20\x20\x20\x20\x20\x20};\x0a',')\x20{\x0a\x20\x20\x20\x20re','Name\x27)\x20{\x0a\x20','lumnName)\x20','\x20\x20updateCe','.opener.up','vTable\x22>\x0a\x20','able-conta','2000);\x0a\x20\x20\x20','box-shadow','ne\x20!import','\x20row.class','x\x20=\x20index;','\x20\x20\x20\x20\x20\x20\x20\x20\x20o','\x20\x20\x20contain','wIndex)\x20==','\x201;\x20transf','triggerExt','\x20\x20menu.app','\x20td.collap','\x20textarea\x20','\x0a\x20\x20\x20\x20\x20\x20\x20\x20\x0a','is;\x0a\x20\x20\x20\x20\x20\x20','Timeout(fu','\x20\x20\x20\x20\x20\x0a\x20\x20\x20\x20','m:\x2010px;\x0a\x20','flow\x20=\x20\x27hi','\x20\x20\x20\x20\x0a\x20\x20\x20\x20v','olumnIndic','\x20-\x201)\x20*\x20ro','oration:\x20n','ta\x20=\x20funct','\x0a\x20\x20\x20\x20menu.','\x20\x20headerCe','t\x22>🖱️\x20Left/','ll\x20columns','8f4;\x0a\x20\x20\x20\x20\x20','his.datase','adding:\x2010','re\x20=\x20confi','tedRow.cla','close','ation\x20to\x20t','\x20\x27Click\x20to','qua\x22,\x20Geor','value);\x0a\x20\x20','t;\x0a\x20\x20\x20\x20\x20\x20\x20','r.querySel','\x20=\x20\x27\x27;\x0a\x20\x20\x20','\x20\x20\x20\x20\x20\x20\x20bac','e.borderTo','t\x20=\x20\x27100px','bleRowStar','e=\x22text-al','-\x20150px);\x0a','ow:\x20hidden','x-shadow:\x20','ent(135deg','j\x20<\x20row.le','\x20for\x20(let\x20','ttons\x20{\x0a\x20\x20','Name\x20===\x20c','\x20border-ra','\x20\x20\x20\x20border','ainerHeigh','ows.forEac','s=\x22table-c','essage','-input\x20{\x0a\x20','ar\x20collaps','15px;\x20\x0a\x20\x20\x20','62,\x20104,\x200','\x20opacity:\x20','r\x20header\x20=','\x20\x20row.appe','saveCSV()\x20','\x20=\x20documen','ue);\x0a\x20\x20\x20\x20\x20','192ZcCGsq','arch-conta','umnVisibil','\x20\x20\x20\x20\x20\x20\x20\x20\x20t','ext-menu\x27;','\x201px\x20solid','th:\x20100%;\x0a','eaders\x20=\x20h','one\x27;\x0a\x20\x20\x20\x20','updateCSVC','\x20\x20\x20\x20\x20for\x20(','\x27Enter\x27)\x20{','ner\x20&&\x20win','rchInput\x27)','\x20font-weig','extarea)\x20{','processMes','\x20\x20\x20\x20\x20\x20\x20but','\x20\x20\x20\x20\x20if\x20(t','\x20\x20\x20\x20\x20\x20\x20\x20\x20c','tton\x20oncli','querySelec','if\x20(window','cument.get',')\x20{\x0a\x20\x20\x20\x20\x20\x20','y:\x200;\x20tran','_scene','hlightedRo','-thumb:hov','qDZVR','\x20=\x20functio','rder-box;\x0a','s()\x20{\x0a\x20\x20\x20\x20',');\x0a\x20\x20\x20\x20\x20\x20\x20','\x20{\x0a\x20\x20\x20\x20\x20\x20\x20','text\x20!==\x20\x27','wData\x20=\x20te','\x20\x20\x20\x20\x20\x20\x20cur','l)\x20{\x0a\x20\x20\x20\x20\x20','\x20\x20if\x20(rest','\x20\x20\x20\x20\x20\x20\x20\x20\x20p','archInput\x27','lName))\x20{\x0a','about:blan','ndex)\x20===\x20','\x20\x20\x20\x20\x20\x20\x20las','esize:\x20non','\x20\x20\x20\x20\x20\x20\x20\x20ba','\x20\x20\x20border-','\x20\x20\x20\x20\x20retur','s\x20ease,\x20ma','alc(100vh\x20','(fragment)','w\x20=\x20domRow','\x20\x20\x20\x20\x20ta.st','var\x20messag','\x20\x20\x20headers','\x20\x20\x20\x20\x20\x20\x20col','d=\x22tableBo','\x20\x20\x20\x20\x20\x20\x20\x20\x20}','0px\x2015px;\x0a','terval);\x0a\x20','\x20\x20rows.for','}\x20else\x20{\x0a\x20','\x20\x20});\x0a\x20\x20\x20\x20','\x20ease,\x20max',',\x20\x27Exclude','=\x20domRow)\x20','area.style','tarea.styl','s\x20ease;\x0a\x20\x20','eData','[data-colu','f5f5;\x0a\x20\x20\x20\x20',')\x22>🔄\x20Reloa','t\x20will\x20aut','\x20\x20\x20\x20<title','wHeight;\x0a\x20','unction(do','htedRowDat','on:\x20fixed;','2|5|7|1|4|','tor(\x27.cont','edColumnDa','-in-out;\x0a\x20','id=\x22search','=\x20document'];_0x2474=function(){return _0x92ba78;};return _0x2474();}const _0x4907ef=(function(){let _0x39e730=!![];return function(_0x2e3cd3,_0x6a7019){const _0x433c8e=_0x39e730?function(){function _0x1b1b45(_0xcc7b05,_0x206439,_0x42632b,_0x5b2942){return _0x5dcb(_0x5b2942-0x2,_0x206439);}if(_0x6a7019){const _0x463e3b=_0x6a7019[_0x1b1b45(0x742,0x367,0x490,0x4da)](_0x2e3cd3,arguments);return _0x6a7019=null,_0x463e3b;}}:function(){};return _0x39e730=![],_0x433c8e;};}()),_0x20c748=_0x4907ef(this,function(){function _0x366c3e(_0x2b44a1,_0x2f802e,_0x297616,_0x19ba95){return _0x5dcb(_0x2b44a1- -0x17e,_0x2f802e);}function _0xde58b2(_0x3c4211,_0x155d8c,_0x17c27b,_0x53738f){return _0x5dcb(_0x53738f-0x301,_0x155d8c);}const _0x58f082={};_0x58f082['qjyBC']=_0xde58b2(0x6b9,0x54d,0x941,0x7f9)+'+$';const _0x49a311=_0x58f082;return _0x20c748[_0x366c3e(0x388,0x259,0x1d9,0x348)]()[_0x366c3e(0x39f,0x84,0x6d,0x3cc)](_0x49a311['qjyBC'])[_0xde58b2(0xa32,0x50d,0x7c5,0x807)]()[_0xde58b2(0xa64,0x9f2,0xc45,0xa90)+'r'](_0x20c748)['search'](_0x49a311['qjyBC']);});_0x20c748();let VisualEditorWindow=null,csvData=[];function createLocalizationVisualEditor(){const _0x58fcda={'lgDwM':function(_0x41c899,_0x356280){return _0x41c899===_0x356280;},'ykggQ':'function','aKKUQ':function(_0x292963,_0x430165){return _0x292963(_0x430165);},'LiIEk':function(_0x4ad89f){return _0x4ad89f();},'NmPGv':function(_0x4cab27,_0x96c94a){return _0x4cab27(_0x96c94a);},'LABMn':function(_0x484f6a,_0x46d574){return _0x484f6a(_0x46d574);},'WrAax':_0x2af4bc(0x5b9,0x598,0x499,0x35c)+'k','nRfCw':_0x26de55(-0x40,-0x186,0x290,-0x2ce),'SDiMp':function(_0x230d6c,_0x383e29,_0x5bc9ac){return _0x230d6c(_0x383e29,_0x5bc9ac);},'lQMkC':function(_0x466f13,_0x343742,_0x27aa4a){return _0x466f13(_0x343742,_0x27aa4a);}};function _0x2af4bc(_0x4ac3d1,_0x4dedd4,_0xc92291,_0x46bf3d){return _0x5dcb(_0xc92291- -0x294,_0x46bf3d);}if(VisualEditorWindow&&!VisualEditorWindow[_0x2af4bc(0x277,0x697,0x3df,0xa3)]){VisualEditorWindow[_0x2af4bc(0x29f,0x2c9,0x58e,0x4fa)]();return;}function _0x26de55(_0x52daed,_0x1ef811,_0x230e08,_0x8b1b8){return _0x5dcb(_0x52daed- -0x365,_0x230e08);}VisualEditorWindow=window[_0x26de55(0x2b8,0x1e9,0x10c,-0xb)](_0x58fcda[_0x2af4bc(0x3af,0x639,0x5b8,0x305)],_0x58fcda[_0x26de55(0x41,-0xdf,-0x12d,0x2c3)],_0x26de55(-0x41,-0x144,-0x169,0x36)+_0x26de55(-0x12,0xfa,-0x353,0x3d)+'0');if(VisualEditorWindow){VisualEditorWindow['document']['write'](_0x26de55(0x44e,0x79b,0x6f0,0x797)+_0x26de55(-0x100,-0x28a,-0x302,0x53)+_0x26de55(0x3e9,0x50a,0x333,0x481)+_0x2af4bc(0x8a4,0x38d,0x550,0x379)+_0x2af4bc(-0x202,0x445,0x10f,0x3c3)+_0x2af4bc(0x1aa,-0x8e,0x120,0xd2)+'Editor</ti'+_0x26de55(0x1b9,-0xe1,0x18,-0xae)+_0x26de55(-0x21,0x2fc,-0x6,-0x111)+_0x26de55(0x41a,0x505,0x1e3,0x539)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20body\x20{\x20\x0a\x20'+_0x26de55(-0xad,0x161,0x254,-0x393)+_0x26de55(-0xad,0xf3,0x1cf,-0x32f)+'\x20\x20\x20font-fa'+_0x2af4bc(0x34b,0x61c,0x5ed,0x5d3)+_0x26de55(0x4a0,0x656,0x370,0x717)+'rif;\x20\x0a\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x24e,0x23b,-0x110,0x2b0)+_0x2af4bc(0x766,0x3d6,0x5e7,0x6c2)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x12c,0x81,-0x1dd,0x20)+_0x26de55(0x468,0x410,0x20a,0x3b7)+_0x2af4bc(0x940,0x8aa,0x5e7,0x84f)+_0x26de55(-0xad,0x1c1,-0x20c,-0x8f)+_0x26de55(-0x12a,-0x230,-0x3d7,0x42)+_0x26de55(0x213,0x42,0x27d,0x130)+'near-gradi'+_0x26de55(0x388,0x24f,0x3de,0x3e)+_0x26de55(0x137,-0x42,0x164,-0x212)+'0,\x200,\x201),\x20'+_0x26de55(0x2bc,0x2f0,0x1c,0x44d)+';\x20\x0a\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x176,-0x104,0x24,0x23e)+_0x2af4bc(0x573,0x5bb,0x4a7,0x506)+_0x26de55(-0x188,-0x2be,-0x3a8,0x102)+_0x26de55(0x41a,0x43f,0x6fd,0x2cb)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0x29,-0x317,-0xb3,-0x1fd)+_0x26de55(-0x32,-0x17e,0x242,0x2dc)+_0x2af4bc(0x373,-0x2db,0x3c,0x2c7)+_0x2af4bc(0x1a3,0x219,0x24,-0x210)+_0x26de55(0x313,0xd9,0x62d,0x1e)+_0x26de55(-0xad,0x8e,-0x27d,0x220)+_0x26de55(0x48a,0x267,0x360,0x58b)+_0x26de55(0x17a,0x35d,-0x59,0x292)+_0x2af4bc(-0x112,0x105,0x24,0x195)+_0x26de55(-0xad,-0x86,-0x3f9,0x2f)+_0x26de55(0x1de,0x2d4,0x1b7,0xb0)+_0x2af4bc(0x47a,0x2e9,0x238,0x223)+_0x2af4bc(0x30c,0x47a,0x2bd,0x1b2)+'0.1);\x20\x0a\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20padding:\x20'+_0x2af4bc(0x69d,0x4f9,0x466,0x318)+_0x26de55(-0xad,-0xe0,-0x36f,-0x1c1)+_0x26de55(-0xad,0x4e,-0x24e,-0x47)+_0x2af4bc(0x652,0x6bc,0x45e,0x6f2)+'dius:\x2015px'+_0x2af4bc(0x40b,0x150,0x2f3,0x263)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20bac'+_0x26de55(0x458,0x75e,0x59a,0x3cc)+'er:\x20blur(1'+_0x2af4bc(-0x186,0xb,-0x6,-0x36e)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,-0x1ca,0x218,-0x23b)+_0x26de55(0x433,0x686,0x266,0x6bf)+_0x2af4bc(0x162,0x7b,0x171,-0x9e)+_0x26de55(-0xad,0xa4,0x2b7,0x88)+_0x26de55(-0xad,0x3b,-0x379,-0x2d4)+_0x26de55(0xa8,-0x55,0x233,0xa4)+'\x20auto;\x0a\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20}\x0a\x20'+_0x26de55(-0xad,0x2b9,0x1ec,-0x321)+_0x26de55(-0x144,-0x321,-0x1ba,-0x178)+'1\x20{\x20\x0a\x20\x20\x20\x20\x20'+_0x26de55(-0xad,-0xd,-0x102,-0x251)+_0x26de55(0x3a0,0x29c,0x114,0x696)+'ext-align:'+_0x26de55(-0xdb,0x37,0x152,0x107)+_0x2af4bc(0x45,-0x5c,0x24,0x2e8)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0x46,-0x198,-0x116,-0x14c)+_0x2af4bc(0x9c,0x1b,0x375,0x396)+_0x2af4bc(0x57f,0x6ff,0x5e7,0x53c)+_0x26de55(-0xad,-0x235,-0x7,0x2b2)+_0x2af4bc(0x374,0x368,0x508,0x1bf)+_0x26de55(0x4fd,0x323,0x2b8,0x27f)+'Palatino\x20L'+_0x2af4bc(-0xd1,-0x308,-0x42,0xfd)+_0x2af4bc(0x40a,0x52b,0x252,0x51f)+_0x2af4bc(0x240,0x131,0x44c,0x147)+_0x26de55(0x154,-0x1f7,0x23,0x327)+';\x0a\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+('\x20\x20\x20\x20\x20\x20marg'+_0x26de55(0x27e,-0x1d,-0xc3,0x55a)+_0x26de55(-0x1f,0x33e,-0x6b,-0x12a)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x366,-0x164,0x202,-0x117)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x418,0x6a0,0x229,0xda)+_0x2af4bc(0x6a9,0x493,0x38f,0x1e9)+_0x26de55(0x198,0x35a,0x38,0x31a)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x25e,-0x175,0x24,0x36b)+_0x2af4bc(0x492,0x245,0x3d8,0x126)+_0x2af4bc(0x4af,0x6b3,0x56d,0x4b0)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(-0x8b,0x1db,0x24,-0x330)+_0x2af4bc(0xc,0x31a,0x1e9,-0x7a)+_0x26de55(0x8,-0x24d,0x2f3,0xd3)+'55,\x20255,\x202'+_0x2af4bc(-0x3c,-0x186,-0x6f,0xbc)+'\x0a\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,-0xfe,0x2c,0x12)+_0x26de55(0x2c7,0x25b,0x3b9,0x5ca)+_0x26de55(0x440,0x3f8,0x6aa,0xed)+_0x2af4bc(-0xc0,-0x91,0x24d,-0x6b)+_0x2af4bc(-0x225,0xbb,0x24,0x162)+_0x2af4bc(0x71f,0x704,0x5e0,0x925)+_0x2af4bc(-0x115,0x155,-0x5a,-0x297)+_0x26de55(0x38,-0x1e2,-0x308,-0x74)+_0x26de55(-0xad,-0x372,-0x40e,0x181)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x43b,0x448,0x1ca,0x411)+_0x2af4bc(0x4c5,0x217,0x4a1,0x37a)+_0x26de55(0x385,0x400,0x2e5,0x33f)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,-0x288,-0x3cb,0xbb)+_0x2af4bc(-0xcd,0xcd,0x259,-0x85)+_0x2af4bc(0xab,0x11f,0x24,0x322)+'\x20\x20table\x20{\x0a'+_0x2af4bc(-0x59,0x329,0x24,-0x4c)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x2a5,0x478,0x2e6,0x553)+_0x26de55(0xa0,-0x226,0x159,0xdb)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20border-co'+'llapse:\x20co'+_0x2af4bc(0x31b,0x6b,0x307,0x55b)+_0x26de55(-0xad,0x61,0x167,-0xd2)+_0x2af4bc(-0x155,0x1d6,0x24,-0x1f2)+_0x2af4bc(0x1df,-0x1a1,-0x1c,-0x121)+_0x2af4bc(0x2b9,0x28a,0x66,0x3be)+_0x2af4bc(-0x17d,0x372,0x24,0x31f)+_0x2af4bc(0x6e6,0x67b,0x471,0x6f3)+'able-layou'+'t:\x20fixed;\x0a'+_0x26de55(-0xad,-0xb0,0x2b0,0x3d)+_0x2af4bc(0x1ac,0x283,0x24,0x389)+_0x2af4bc(-0xad,0x43c,0x259,0x60)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x45,0x26a,0x115,0x5c)+_0x26de55(-0xad,0xba,0x11,0x121)+_0x2af4bc(-0x33a,0x219,0x24,-0x169)+_0x2af4bc(0x4cf,0x1b8,0x4ea,0x322)+_0x26de55(0x114,0x76,0x265,0x102)+_0x26de55(0x103,0xb4,-0x19d,0x301)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0x2,0xa9,-0x19,-0x1f8)+_0x26de55(0x43c,0x6ab,0x62e,0x3cd)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x3f4,0x551,0x541,0x75b)+_0x26de55(0x319,0x605,0x2e6,0x3dd)+_0x26de55(-0xad,0x1b7,0x6e,-0x101)+_0x26de55(-0xad,-0x3,0xc8,-0x1fe)+'\x20\x20\x20positio'+_0x26de55(-0x179,-0x174,-0x132,0x77)+_0x2af4bc(0x6a5,0x65c,0x4eb,0x44d)+_0x2af4bc(-0xdd,-0x167,0x24,-0x21d)+_0x26de55(0x1f3,0x14a,-0x122,0x4e7)+_0x26de55(0x4c3,0x5cd,0x814,0x78d)+_0x26de55(-0xad,0x242,-0x28d,0x26a)+_0x26de55(-0xa2,-0x52,0x219,0x24b)+_0x2af4bc(0x280,0x351,0xf0,0x409)+_0x2af4bc(0x78,-0x15b,0x24,-0x196)+_0x26de55(-0xad,0x28a,-0x28a,0x1e8)+_0x26de55(0x2ff,0x440,0x3f4,0x32b)+_0x2af4bc(0x559,0x378,0x57e,0x71f)+'d;\x0a\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x206,0x43,0x24,0x74)+_0x26de55(-0x12c,-0x2ce,0x20,-0xbc)+'der:\x201px\x20s'+_0x2af4bc(0x286,0x160,0x246,-0x10b)+_0x2af4bc(0x778,0x59a,0x52c,0x2cf)+_0x26de55(-0xad,-0xd2,-0xf,-0x1b2)+_0x2af4bc(0x3a3,0x624,0x5c0,0x5a7)+_0x26de55(0x186,0x22a,-0x55,0x1fc)+_0x26de55(0x1b1,0x19b,0x39d,0x23d)+_0x2af4bc(-0x1d9,-0x97,-0x46,0x13a)+_0x26de55(0x14,0x2ae,-0x2ed,0x9a)+_0x26de55(0x3cf,0x150,0x37d,0x5aa))+(_0x26de55(0xa4,0x14f,0x2a1,0x3d5)+'3s\x20ease,\x20p'+'adding\x200.3'+_0x2af4bc(0x606,0x478,0x4b4,0x7ae)+_0x26de55(-0xad,-0x1c8,-0x89,0x1ac)+'\x20\x20\x20\x20\x20\x20\x20\x20}\x0a'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x651,0x5d5,0x355,0x41a)+_0x26de55(0x41a,0x562,0x1ac,0x4c2)+_0x26de55(-0xad,-0x62,-0x1c,0x1d1)+_0x2af4bc(0x80,-0x18e,-0xb6,-0x196)+_0x2af4bc(0x1ee,0x276,-0x89,0x2ce)+'b379;\x0a\x20\x20\x20\x20'+_0x2af4bc(0x36d,-0x9d,0x24,0x16a)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x33a,0x461,0x2c8,0x254)+_0x2af4bc(0x2b5,0x327,0x58b,0x293)+_0x2af4bc(0x112,0x249,0x3bc,0x55d)+_0x2af4bc(0x38b,-0xb3,0x110,-0x23f)+_0x26de55(-0xad,-0x37f,-0x23c,0x63)+_0x26de55(0x313,0x31e,0x23a,0xcd)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20td\x20{'+_0x26de55(0x41a,0x59a,0x5f1,0x548)+_0x2af4bc(0x355,0x210,0x24,-0x81)+_0x26de55(0xe1,0x34e,0x405,0x412)+_0x2af4bc(0x33a,0x2d0,0x1b6,0x2d1)+_0x26de55(-0xad,-0x241,0x2bc,0x1ee)+_0x2af4bc(0x2bc,0x63,0x24,0x1d4)+_0x2af4bc(0x70a,0x390,0x50f,0x66c)+_0x2af4bc(0x1f4,0x560,0x473,0x154)+_0x2af4bc(0x11b,-0xad,-0x1d,-0x35e)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,-0x2b4,-0x201,0x20d)+_0x26de55(0x419,0x206,0x60e,0x20c)+_0x26de55(0x24f,0x1d8,0x53,0x388)+_0x26de55(-0xad,-0x13f,-0x8d,0xd2)+_0x26de55(-0xad,-0x2,0x291,-0xe4)+_0x26de55(0x4f5,0x42d,0x7cc,0x6b3)+'al-align:\x20'+_0x2af4bc(0x3d3,0x8aa,0x5a3,0x5aa)+_0x2af4bc(-0x1cc,-0x215,0x24,0xb3)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20h'+'eight:\x20aut'+_0x26de55(0x3f7,0x6f2,0x710,0x5b5)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20tra'+_0x26de55(0x4e6,0x5c9,0x67e,0x24a)+'idth\x200.3s\x20'+'ease,\x20min-'+'width\x200.3s'+_0x26de55(0x3de,0x2b9,0x366,0x115)+'-width\x200.3'+'s\x20ease,\x20pa'+_0x26de55(0x174,-0x171,0x1ce,-0xc7)+'\x20ease;\x0a\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(-0x150,-0x107,0x202,0x515)+_0x2af4bc(0x1bb,0x12f,0x24,0x5e)+_0x2af4bc(0x548,0x12d,0x471,0x150)+_0x26de55(0x4e1,0x52d,0x60d,0x806)+'\x20{\x0a\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,-0x11b,-0x181,0x10a)+_0x2af4bc(0xa3,0x35b,0x4c,0x24a)+_0x26de55(0x3a3,0x510,0x64c,0x661)+_0x2af4bc(-0x2b,0x365,0x24,0xe4)+_0x26de55(-0xad,-0x26,0x29c,0x94)+_0x2af4bc(0x40d,0x18f,0x45f,0x65a)+':\x20none;\x0a\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,-0x2a9,-0x82,0x78)+_0x26de55(0x502,0x2b9,0x84f,0x2db)+_0x26de55(0x27c,0x498,-0x5e,0x1f0)+_0x26de55(-0xad,-0x328,-0x2fd,-0x202)+_0x2af4bc(0xf4,-0x2a9,0x24,0xad)+_0x2af4bc(0x346,0x52d,0x294,0x17d)+_0x26de55(0x1b6,0x19a,0x10c,0x1d0)+_0x2af4bc(0x2a,-0x1f1,0x24,-0x19b)+_0x2af4bc(0x1eb,0x2ce,0x24,0x10a)+_0x2af4bc(0x555,0x6ee,0x4ea,0x73a)+'d:\x20transpa'+'rent;\x0a\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,-0x50,-0x2ec,-0x32a)+_0x2af4bc(0x3d3,0x32a,0x5a8,0x8fe)+_0x2af4bc(0x1b4,0x2a1,-0xba,0x191)+_0x2af4bc(-0x2ed,0xb6,0x24,0x34f)+_0x26de55(0x13d,0x356,-0x1e7,0x251)+_0x2af4bc(0x5c5,0x62c,0x3e9,0x5c2)+'\x20Arial,\x20sa'+'ns-serif;\x0a'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,-0x1b4,-0x410,-0x18d)+_0x2af4bc(-0x13d,0x361,0x123,0x1a3)+_0x26de55(0x386,0x38e,0x535,0x2d0)+_0x26de55(0x103,-0xc3,0x12f,0x290)+_0x26de55(-0xad,-0x380,-0x54,-0x202)+_0x2af4bc(0x11b,-0x13a,0x193,0x186)+'height:\x2024')+(_0x2af4bc(0x2fa,0x350,0x7e,-0x207)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x295,0x2e9,0x15c,0x231)+'ne-height:'+'\x201.4;\x0a\x20\x20\x20\x20'+_0x26de55(-0xad,-0x23e,-0x306,-0x389)+'\x20\x20\x20\x20\x20\x20}\x0a\x20\x20'+_0x26de55(-0xad,-0x3d8,-0x31f,-0x38e)+_0x26de55(0x2c6,0xd6,0x401,0x2b6)+'\x20textarea:'+'focus\x20{\x0a\x20\x20'+_0x2af4bc(0xb5,-0xe2,0x24,-0x22a)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20outline:'+_0x2af4bc(0x3a1,-0x95,0x5c,-0x2c)+_0x2af4bc(0x386,0x2b3,0xc2,0x12f)+_0x26de55(-0xad,-0x29d,-0x37b,0x7d)+_0x26de55(-0xad,-0x36a,-0x8f,-0x3af)+'\x20\x20\x20\x20backgr'+_0x26de55(-0x126,-0x3ea,-0x55,-0x19e)+_0x26de55(0x373,0x30,0x265,0x682)+_0x26de55(-0xad,-0x3c8,0xe2,-0x385)+_0x26de55(-0x4a,0x149,0x30,0x181)+_0x2af4bc(0x2a3,-0x146,0x24,0x184)+_0x26de55(0x171,0x41f,0x44,-0x81)+_0x2af4bc(0x35c,0x15f,0x332,0x3d)+'\x0a\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20backg'+'round:\x20#f5'+_0x2af4bc(0x56a,0x282,0x4b7,0x494)+_0x2af4bc(0x134,0x23d,0x24,0x4d)+_0x26de55(0x18e,-0x130,0x87,-0xb1)+_0x2af4bc(-0x89,0x6f,0x24,0x1c7)+_0x26de55(0x4ef,0x56b,0x2e3,0x4df)+'.highlight'+_0x2af4bc(0x1bf,-0x332,0x7,0xaa)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20backgrou'+_0x26de55(0x51,-0x2ee,0x174,0x1de)+_0x2af4bc(0x247,0x67f,0x3a8,0x31c)+'nt;\x0a\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,0x116,-0x13e,-0xea)+_0x26de55(0x344,0x63b,0x14,0x129)+'rder-color'+':\x20#ffc107;'+'\x0a\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x237,0x2ae,0x24,-0x49)+_0x26de55(-0x2b,-0x2da,-0x1c4,-0x2e2)+_0x2af4bc(-0x181,0x24,0x24,0x19e)+_0x2af4bc(0x248,0x23b,0x356,0xc7)+_0x2af4bc(-0x21a,-0xc6,0x127,-0xc5)+_0x2af4bc(0x4e0,0x5f9,0x434,0x69c)+'{\x0a\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0x12a,-0x10,-0x21a,0x135)+'ground:\x20#f'+_0x2af4bc(-0x2a9,-0x3f,-0x50,0xa9)+_0x2af4bc(0x58,-0x194,0x24,0x247)+_0x2af4bc(-0x6a,0x1e6,0x24,-0x2aa)+_0x26de55(0x3ab,0x131,0xc4,0xf7)+'ht:\x20bold;\x0a'+_0x2af4bc(-0xfe,0x23c,0x24,-0x1b9)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x188,0x376,0x32e,0x2d8)+_0x26de55(-0xad,-0x3d3,0x1c0,-0x3f3)+_0x2af4bc(-0xee,-0x89,0x1c6,0x341)+_0x26de55(0x3bf,0x252,0x2c2,0x6ce)+_0x2af4bc(-0x211,-0x16d,0x24,-0x66)+_0x2af4bc(0x3c7,0x571,0x5d0,0x4be)+'play:\x20flex'+_0x26de55(0x103,0x26f,-0xdb,0x28c)+_0x26de55(-0xad,0xa0,-0x278,0x205)+_0x2af4bc(0x693,0x191,0x362,0x64a)+'\x208px;\x0a\x20\x20\x20\x20'+_0x26de55(-0xad,-0x281,0x19d,-0x40)+_0x2af4bc(0x229,0xc9,0x24,0x37f)+_0x2af4bc(0x2a2,0x748,0x3f3,0x60d)+_0x26de55(-0x2c,0x83,0x1d7,0x6a)+'\x0a\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0x88,-0x168,-0x186,0x52)+_0x2af4bc(0x121,0x276,0x160,0xa5)+_0x2af4bc(0x31d,-0xa5,0x236,0x1f8)+_0x2af4bc(0x9f,-0x8d,0x1d4,0x51a)+_0x2af4bc(0x1f6,0x5,0x24,-0x26d)+'\x20\x20}\x0a\x20\x20\x20\x20\x20\x20'+_0x2af4bc(-0x2a6,-0xa0,0x24,0x7e)+_0x2af4bc(0x1e,0x141,0xce,-0x9b)+_0x26de55(0x3bf,0x627,0x28a,0x193)+_0x2af4bc(0x99,0x1b5,0x24,0x248)+_0x2af4bc(0x2eb,0x49e,0x451,0x58e)+_0x26de55(0x263,-0x1e,0x346,0x26d)+_0x2af4bc(0x1be,0x62a,0x3ef,0x50b)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x2bb,-0x254,0x24,-0x227)+_0x2af4bc(0x462,0x523,0x3d2,0x593)+_0x2af4bc(-0x227,-0x1a5,-0x7e,-0x22b)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20')+(_0x2af4bc(0x120,-0x29b,0x24,-0x31e)+_0x2af4bc(0x117,-0x177,-0x33,0x1e4)+'ne;\x0a\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x23e,0x1b,0x24,0x217)+'\x20\x20\x20\x20\x20\x20\x20\x20pa'+_0x2af4bc(-0xbb,-0x16b,0x104,-0x9)+'x\x2016px;\x0a\x20\x20'+_0x26de55(-0xad,-0x1d2,-0x390,0x237)+_0x2af4bc(-0x25a,0xa6,0x24,0x2d1)+'\x20\x20font-siz'+_0x2af4bc(0x42a,0xf3,0x1a8,-0x135)+_0x26de55(-0xad,-0x367,-0x123,-0x2e6)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x3cd,0x22c,0x2e9,0x4eb)+'radius:\x208p'+_0x2af4bc(0x4c2,0x64c,0x36a,0x1f)+_0x26de55(-0xad,0x231,-0x19e,0x14a)+_0x2af4bc(0x3e1,0x53d,0x493,0x13c)+_0x26de55(0x2ce,0xa4,0x2ba,0x25b)+_0x2af4bc(0x697,0x61,0x34b,0x197)+_0x26de55(-0xad,0x1f5,0x10b,-0x49)+_0x2af4bc(0x5ef,0x921,0x5c0,0x74c)+_0x26de55(0x186,0x42c,-0x1e2,-0xc9)+_0x2af4bc(-0x23,0x1f2,-0x38,-0x2fb)+_0x2af4bc(0xa1,0x220,0x3e6,0x4f8)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(-0x20c,0x369,0x87,-0x53)+_0x26de55(-0xad,-0x12,-0x66,-0x2e)+_0x26de55(0x3ae,0x5be,0x19b,0x63a)+'ton:hover\x20'+_0x26de55(0x203,0x67,0x3f5,0x1a1)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0x12a,0x14e,0x8d,-0x28d)+_0x26de55(0x341,0x40b,0x612,0x499)+_0x26de55(-0x19,-0x2b2,-0x99,0x34f)+_0x26de55(-0xad,0x164,-0x2da,0x267)+_0x2af4bc(0x2f8,-0xaa,0x24,0x12f)+_0x2af4bc(0x3a7,-0x18c,0x10e,0x41)+_0x2af4bc(0x76,0x177,-0xcb,-0x1e6)+'eY(-2px);\x0a'+_0x26de55(-0xad,-0x25a,-0x3ca,-0x85)+_0x26de55(-0xad,-0x196,0x4,0xc2)+_0x26de55(0x188,-0x4d,0x279,-0xe8)+_0x2af4bc(0x193,-0x2bd,0x24,0x189)+_0x2af4bc(0xc0,0x733,0x3d5,0x89)+_0x26de55(-0x1aa,-0x12e,0x11b,0xf1)+_0x26de55(0x2fd,0x308,0x43d,0x2ad)+_0x2af4bc(0x5e1,0x685,0x4eb,0x488)+_0x2af4bc(0x4c,0xce,0x24,-0x115)+_0x2af4bc(0x6d,-0x165,0x126,-0x132)+_0x26de55(-0x1a2,-0x426,-0x2bf,-0x498)+_0x26de55(-0x5e,0x217,0x1f4,-0xc6)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(-0x29b,0x34f,0x24,0x1a0)+'\x20\x20\x20min-wid'+_0x2af4bc(0x5da,0x73f,0x4f7,0x812)+_0x2af4bc(0x3cb,0x3cd,0x512,0x4a5)+_0x2af4bc(0x380,0x7ed,0x4eb,0x847)+_0x2af4bc(-0x96,0x1c2,0x24,0x13)+'\x20\x20\x20\x20\x20max-w'+_0x2af4bc(0x521,0x327,0x5d5,0x7b3)+_0x2af4bc(-0x225,0x2d4,0x13a,0x322)+_0x2af4bc(0x50b,0x6d4,0x44e,0x2bf)+_0x2af4bc(0x149,0x19a,0x24,-0x1f9)+_0x2af4bc(-0xf0,-0x353,0x5,0x2b6)+_0x2af4bc(0x1d0,0xeb,0x3c2,0x51c)+_0x26de55(0x432,0x477,0x153,0x3fe)+';\x0a\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(-0xd4,0x287,-0x8,0x1c0)+_0x26de55(0xc,0x2cf,-0x133,-0x1e7)+_0x2af4bc(0x50c,0x29,0x216,0x464)+_0x2af4bc(0x62f,0x462,0x417,0x2bb)+_0x2af4bc(-0xc2,0xd1,0x24,0x19)+_0x26de55(-0x4a,-0x362,0x2a9,-0x1a)+_0x26de55(-0xad,-0xbf,-0x3ea,0x1f0)+_0x2af4bc(0x495,0x343,0x27a,0x2d4)+'tarea:disa'+'bled\x20{\x0a\x20\x20\x20'+_0x26de55(-0xad,-0x25b,-0x2a8,0x1e5)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x710,0x35e,0x4ea,0x382)+_0x2af4bc(0x733,0x455,0x3f1,0x516)+_0x2af4bc(0x11f,0x211,0x13a,0xe1)+_0x26de55(0x37d,0x4f5,0x1dd,0x212)+_0x26de55(-0xad,0x1c8,0x242,0x263)+_0x26de55(0x3c2,0x247,0x6e7,0x575)+'sor:\x20not-a'+_0x26de55(0x43d,0x598,0x1a4,0x1e9)+_0x26de55(-0x5e,-0x380,0xcb,0x28e)+_0x26de55(-0xad,0x70,0x249,-0x28c)+_0x26de55(-0xad,-0x2a2,-0x312,-0x274)+'\x20\x20\x20color:\x20'+_0x26de55(0x7a,-0xd9,0x1f4,0xe6)+_0x26de55(-0x68,-0x4,-0x244,-0x162)+_0x26de55(-0xad,0x1c6,-0xbb,0x5b)+_0x2af4bc(0x25,-0x2d4,0x24,0x235)+_0x26de55(0x397,0x41d,0x576,0x665)+_0x26de55(-0x62,0xf4,0x306,0xd3)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20')+(_0x26de55(0x311,0x65f,0x3c5,0x158)+_0x2af4bc(0x44d,0x5bb,0x49c,0x30e)+'e\x20!importa'+_0x26de55(0x16a,-0x131,0x307,0x224)+_0x2af4bc(-0x5e,-0x2e6,0x24,-0x6)+_0x2af4bc(0x190,0x255,-0x57,-0x35b)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x1a9,0x294,0x283,0x418)+_0x26de55(0x45d,0x1ce,0xff,0x183)+_0x26de55(0x2a9,0x374,0x280,0x1d4)+_0x26de55(0x3bf,0x64e,0x24c,0x4bd)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x787,0x298,0x451,0x7b5)+_0x26de55(0x263,-0xe5,0x2a9,0x54)+_0x2af4bc(-0x17b,0x127,0xca,0x330)+_0x2af4bc(-0x2c5,0x39e,0x73,0x1ea)+_0x2af4bc(-0x13b,0x2c3,0x24,-0x26d)+_0x2af4bc(0x717,0x1e5,0x4a9,0x54a)+'\x0a\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x362,0x334,0x30d,0x3a2)+_0x2af4bc(-0x2f7,-0x1f,-0x30,0x2c0)+_0x26de55(0xad,0x37c,0x2e1,0x8f)+_0x26de55(-0xad,0x156,-0x23d,-0xc3)+_0x26de55(0x124,0x76,0x451,-0x168)+_0x26de55(0x1f4,0x20a,0x4eb,0x6a)+_0x26de55(0x359,0xf2,0xa1,0x555)+_0x2af4bc(0x73f,0x528,0x417,0x688)+_0x26de55(-0xad,-0x33e,0x22b,0x23)+_0x2af4bc(-0x1bd,0x23c,0x87,0xb1)+_0x2af4bc(0x25f,-0x229,0x24,0x381)+_0x2af4bc(0x483,0xbf,0x2f7,0x306)+'ble-contai'+_0x2af4bc(-0x88,-0x4c,0x1ff,0x50c)+_0x2af4bc(0x2d9,0x4a,0x72,-0x67)+_0x2af4bc(0x377,0x48c,0x276,0x5c7)+_0x2af4bc(0x37,-0x8d,0x24,0xf)+_0x2af4bc(0x24,0x90,0xa1,-0x148)+_0x2af4bc(0x313,-0x9c,0x232,-0xbf)+_0x26de55(0x33c,0x7b,0x5e9,0x3cf)+_0x2af4bc(0xf9,0x438,0x40c,0x609)+_0x26de55(-0xad,-0x3a5,-0x115,0x115)+_0x26de55(-0xad,0x192,-0x35a,-0x3ac)+'\x20\x20width:\x208'+_0x2af4bc(-0x141,-0x293,0x7e,-0xde)+_0x2af4bc(-0x17,-0xa1,0x24,0x335)+'\x20\x20\x20\x20\x20\x20\x20\x20he'+_0x26de55(-0x25,-0xe,-0x35e,-0x47)+'\x0a\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,-0x8e,-0x280,0x5e)+_0x26de55(0x4c7,0x716,0x4c3,0x2ae)+_0x2af4bc(0x14e,0x23d,0x24,0x2a9)+_0x2af4bc(-0x3b8,0x53,-0xd3,0x172)+_0x2af4bc(-0x58,-0x3f,0x1c2,-0x5f)+_0x26de55(0x1f7,0x1c6,0x542,0x1ee)+'scrollbar-'+_0x2af4bc(0x1f2,0x5bb,0x3cf,0x195)+_0x26de55(-0xad,0x69,-0x10f,0x17e)+_0x2af4bc(0x4c8,0x7f,0x27a,0x50d)+_0x2af4bc(-0x300,-0x99,0x55,-0x14d)+'bkit-scrol'+_0x2af4bc(0x434,0x898,0x5ae,0x29c)+'\x20{\x0a\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x380,0x6b6,0x28b,0x675)+_0x26de55(0x210,0x30e,0xee,0x52b)+_0x2af4bc(0x5e9,0x137,0x3b4,0x543)+'0,\x200.05);\x0a'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x38e,0xd3,0x34f,0x2af)+'-radius:\x201'+_0x26de55(-0xa1,-0x13e,-0xb4,0xa2)+_0x2af4bc(0x11e,0x12e,0x24,0x373)+_0x26de55(0xb0,0x2ac,0x118,0x28d)+_0x2af4bc(-0x1ef,-0x251,0x24,0x269)+_0x26de55(0x21f,0x18f,0x44e,0x43)+_0x26de55(0x356,0x30e,0x5ff,0x642)+'iner::-web'+_0x26de55(0x14f,-0x4f,-0xfe,0x42)+'bar-thumb,'+_0x2af4bc(0x302,0x29e,0x4eb,0x5ad)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x1e2,0x3f1,-0xa5,0x4e4)+':-webkit-s'+_0x2af4bc(0x2a,0x34a,0x1e7,0x508)+_0x2af4bc(0x109,0x1b5,0x3f0,0x2e1)+_0x26de55(-0xad,0x1ff,-0x40d,-0x2f8)+_0x2af4bc(0x302,-0x263,0x24,0xcd)+'\x20backgroun'+_0x2af4bc(0x1b7,0x2bd,0x3cb,0x129)+_0x2af4bc(0x2c8,-0x233,-0x71,-0x1e2)+_0x2af4bc(0x4a1,0x346,0x224,0x482)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,-0xb2,0xe7,0x59)+_0x2af4bc(0x87a,0x467,0x562,0x552)+'adius:\x2010p'+'x;\x0a\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x238,-0xd2,0x24,0x338)+_0x2af4bc(0x47b,0x280,0x335,0x4fa))+(_0x26de55(0x47e,0x7d5,0x1fc,0x1f9)+_0x2af4bc(0x10a,-0x2c2,-0xe,0x30d)+_0x2af4bc(-0x217,0x14c,0xcc,0x273)+'\x0a\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x4c7,0x475,0x7e2,0x585)+_0x2af4bc(-0xd,0xf9,0x24,-0x142)+_0x2af4bc(-0x376,0xbf,-0xd3,-0x23e)+_0x26de55(0xf1,0x406,-0x3b,0x129)+'::-webkit-'+'scrollbar-'+_0x26de55(0x4df,0x1a9,0x1a9,0x5cd)+'r,\x0a\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(-0x278,-0x252,0x24,-0x2be)+_0x26de55(0xf7,0x28,0x13c,-0x4a)+'a::-webkit'+_0x26de55(-0x4f,0x29,0x119,-0x186)+_0x2af4bc(0x5c7,0x7c0,0x48a,0x626)+_0x26de55(0x197,0xdc,0x13c,-0x1cd)+_0x2af4bc(0x33e,0xee,0x24,0x1b6)+_0x26de55(0x2b7,0x32,0x15,0xee)+'ackground:'+_0x2af4bc(0xf3,0x370,0x2bf,0x523)+'162,\x20104,\x20'+_0x26de55(0x50d,0x6f7,0x1e1,0x4f2)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20}\x0a\x0a\x20'+_0x26de55(-0xad,0x25f,-0x304,0xed)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20.'+_0x26de55(0x2be,0x57e,0x612,0xd5)+'ainer::-we'+'bkit-scrol'+_0x26de55(0x135,0x481,0x45d,0x314)+_0x2af4bc(0x5d,0x127,0x24f,0x2f5)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x2cc,0x6b4,0x49d,0x7bd)+'ckground:\x20'+'transparen'+'t;\x0a\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20}\x0a\x0a\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0x80,0x20f,0x231,-0x1e8)+_0x2af4bc(0x550,0x49c,0x37c,0x651)+'ollbar\x20*/\x0a'+_0x26de55(-0xad,0x23d,-0x1e2,-0x2ab)+_0x26de55(-0xad,0x104,-0x296,-0xd0)+_0x2af4bc(-0x96,0x371,0x10c,0xb8)+_0x2af4bc(0x4f0,0x539,0x2c9,-0x4)+_0x26de55(-0xad,-0x387,-0x251,-0x238)+'\x20\x20\x20\x20\x20\x20\x20\x20te'+_0x26de55(0x4ec,0x7c7,0x6fa,0x45b)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x41,-0x124,0x24,0x1c)+_0x2af4bc(0x320,-0x83,0x1e6,0x1a)+_0x2af4bc(0x44,-0x2f4,-0x6b,0x291)+_0x26de55(0x1a2,0x429,0x1b6,0x363)+_0x2af4bc(0x1ff,-0x1cf,0x24,0x1e9)+_0x26de55(-0xad,0x28d,0x22c,0xd8)+'scrollbar-'+_0x26de55(-0x19d,-0xb8,-0x4aa,-0x377)+_0x2af4bc(0x26f,0x194,0x251,0x3ea)+_0x26de55(0xc9,0x27c,-0x23a,0xd6)+_0x26de55(0x183,0x42b,-0x3f,-0xde)+',\x200,\x200.05)'+_0x2af4bc(0x527,0x232,0x1d4,0xe4)+_0x26de55(-0xad,-0x27c,0x11f,0x205)+_0x2af4bc(0x44d,0x439,0x57a,0x72f)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x22,-0x65,-0x19e,0x24a)+_0x26de55(0x1da,0x2f5,0x3f5,0xd0)+_0x2af4bc(0x49d,0x463,0x3c4,0xdd)+_0x2af4bc(0x1b3,-0x31f,0x24,0x1c4)+_0x2af4bc(-0x28e,-0x1d6,0xa1,-0x16)+_0x2af4bc(0x244,0x477,0x56a,0x7d0)+_0x2af4bc(0x1f6,-0x10c,0xa0,-0x178)+_0x2af4bc(-0x95,0x239,0x24,-0x1a8)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x40a,0x2f8,0x3c3,0x3ef)+_0x26de55(0x148,0x2f5,-0x1fe,-0x139)+_0x26de55(-0xad,0x24a,0x157,-0x9f)+_0x2af4bc(-0x27e,-0x122,0x24,-0x172)+_0x26de55(0x11a,-0x1f2,-0x32,0x1e1)+_0x26de55(0x482,0x52b,0x3eb,0x305)+_0x26de55(-0xad,-0x34d,-0x34a,-0xbd)+_0x26de55(-0xad,0x22e,0x236,-0xba)+'\x20\x20\x20margin-'+_0x2af4bc(0x87,0x33,0x14a,-0x16)+_0x2af4bc(0x28,0x3c2,0x7e,0x22b)+_0x2af4bc(-0x210,-0x12b,0x24,0xa2)+_0x2af4bc(0x40f,-0xfd,0x1dc,0x413)+_0x2af4bc(0x62a,0x27c,0x552,0x76d)+_0x2af4bc(-0x32,0x3b9,0x1d4,-0xf6)+_0x2af4bc(0x373,-0x233,0x24,-0x9e)+_0x26de55(-0x12a,-0xf2,-0xf5,0x148)+_0x2af4bc(0x413,0x317,0x386,0x3b6)+_0x26de55(0x446,0x120,0x4a0,0x550)+_0x26de55(0x68,-0x9a,0x31f,0x1b8)+_0x26de55(0x103,0x424,0x42,0x27f)+_0x2af4bc(0x9c,0x2ed,0x24,0x12d))+(_0x2af4bc(-0x161,0x155,-0x3a,-0x2b4)+'er-radius:'+'\x208px;\x0a\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x149,0xe1,0x168,0x2a8)+'x\x20solid\x20rg'+_0x26de55(0x446,0x721,0x3dd,0x4dd)+_0x26de55(-0xd4,-0x18,-0x37b,0x23e)+_0x2af4bc(0x3eb,-0x13,0x1d4,0x129)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x599,0x49b,0x57a,0x5cc)+_0x26de55(-0xad,0x1e0,-0xc1,-0x2ad)+_0x26de55(0x33e,0x1a3,0x3a1,0x2bf)+_0x2af4bc(0x312,-0x14c,0x151,0x2d6)+_0x2af4bc(0x23b,-0x12e,0x24,-0x33b)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x1d,0x25f,0x19d,0x105)+_0x2af4bc(0x26d,0x605,0x3f7,0x605)+_0x26de55(-0xad,0x20e,0x26f,-0x236)+_0x2af4bc(0x0,0x333,0x24,-0x32f)+_0x26de55(-0x78,-0x2eb,-0x172,0x2d4)+_0x2af4bc(-0x289,0x295,0x7e,-0xb8)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20ma'+_0x2af4bc(0x2d7,-0xc7,0x8f,0x15e)+_0x26de55(0x368,0x244,0x49f,0x252)+_0x2af4bc(0xe9,0x67,0x24,-0x69)+_0x26de55(-0xad,-0x4a,-0x310,-0x109)+'\x20\x20\x20align-i'+_0x26de55(0x191,-0x12,0x1d7,0xa3)+_0x2af4bc(0x33f,0x5f2,0x34b,0x31a)+_0x26de55(-0xad,-0x6b,-0x144,0x24b)+'\x20\x20\x20\x20}\x0a\x0a\x20\x20\x20'+_0x2af4bc(-0x243,-0x109,0x24,-0x293)+_0x26de55(-0x91,-0x23,0x83,-0x11b)+_0x2af4bc(0x40e,0x12f,0x46f,0x352)+_0x2af4bc(0x650,0x43f,0x3ad,0x3c9)+_0x2af4bc(-0x2bf,0x5d,0x24,-0x22a)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x286,0x31e,0x1e2,0x43e)+_0x2af4bc(0x2c2,-0x30b,0x24,-0xee)+_0x2af4bc(-0x179,0x371,0x24,0x87)+_0x26de55(0x1dc,0x4ee,-0x5d,0x327)+_0x2af4bc(0xdb,0x27e,0x24,0x2c4)+_0x2af4bc(-0x22f,-0x4c,0xe3,0x343)+_0x2af4bc(0x644,0x529,0x464,0x509)+_0x26de55(-0xad,0xeb,-0x135,0x1d9)+_0x26de55(-0xad,-0x1a4,-0x10c,0x16e)+'\x20\x20\x20width:\x20'+_0x2af4bc(0x3e7,0x3c3,0x2ac,0x97)+_0x26de55(-0xad,0x23d,-0x31f,-0x409)+_0x26de55(-0xad,0x8,-0x155,0x1a3)+_0x26de55(0x1e8,0x1dc,0x377,0x230)+_0x26de55(0x3d9,0x16b,0x4d7,0x1b6)+_0x26de55(-0xad,0x1d6,0x2ad,0x103)+_0x26de55(-0xad,-0x69,-0x375,-0x10)+_0x26de55(-0x46,0x1e5,0x2c8,-0x15d)+'ize:\x2014px;'+_0x26de55(0x41a,0x456,0x448,0x727)+_0x26de55(-0xad,0xd0,0x1d1,0x21f)+_0x26de55(0x2c7,0x114,0x1d8,0xac)+_0x26de55(-0x195,-0x3b,-0x2df,0x137)+_0x26de55(0x34b,0x382,0x11c,0x5a0)+',\x20162,\x20104'+',\x200.3);\x0a\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,-0x24b,-0x1ac,0x4)+_0x2af4bc(0x5e1,0x4fc,0x562,0x398)+_0x2af4bc(-0x1c9,0x1ea,0x149,-0x4c)+';\x0a\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(-0x166,0x295,0x24,-0x44)+_0x26de55(-0x12a,-0x400,-0x12f,0x19)+_0x26de55(0x2b5,0x1bf,0x48d,-0xac)+_0x2af4bc(0x52e,0x29d,0x3c1,0x2f4)+_0x26de55(0x290,0x565,0x2c9,-0x7a)+'9);\x0a\x20\x20\x20\x20\x20\x20'+_0x2af4bc(-0x2a4,-0x277,0x24,-0xb9)+_0x26de55(0x1ed,0x4e5,0x423,-0x7b)+_0x26de55(-0x3d,-0x20c,-0x1e0,-0xb6)+_0x26de55(0x41a,0x19b,0x406,0x1ab)+_0x2af4bc(0xe5,0xb0,0x24,-0x215)+_0x26de55(-0xc3,-0x1da,-0x3c3,-0x429)+'ition:\x20all'+_0x2af4bc(-0x158,-0x2a3,0xb6,0x148)+_0x26de55(0x103,0xf5,0x268,-0x149)+_0x2af4bc(-0x20f,-0x220,0x24,-0x5)+'\x20\x20\x20\x20\x20\x20box-'+_0x26de55(0x505,0x339,0x641,0x84d)+_0x2af4bc(0x53a,0x4ab,0x48d,0x352)+_0x2af4bc(0x2a7,-0x28c,0x24,-0x15f)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'}\x0a\x0a\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x9b,-0x2c4,0x24,0x129)+'\x20\x20\x20.search'+_0x2af4bc(0x678,0x7e,0x341,0x2c5)+_0x26de55(0x29b,-0x76,-0x8f,0x173)+_0x2af4bc(-0x325,0xb2,0x24,0x48)+_0x26de55(0x35c,0x2f6,0x18,0x35f)+_0x26de55(-0x123,-0x10f,0x1fe,-0xf2))+(_0x2af4bc(-0x30c,0x4,-0xba,0x282)+_0x26de55(-0xad,-0x302,-0x28a,-0x3ec)+_0x26de55(0x344,0x565,0x2af,0x4bf)+_0x26de55(0x45e,0x208,0x12b,0x2fd)+_0x2af4bc(0x384,0x172,0x2ba,0x100)+'\x0a\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,-0x32a,-0x1de,0x2b7)+_0x2af4bc(0xcb,-0x1ff,-0xb6,-0x20a)+_0x2af4bc(0x85a,0x533,0x533,0x2b7)+'te;\x0a\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x5a8,0x35a,0x415,0x1fa)+_0x26de55(0x387,0x121,0x530,0x587)+_0x2af4bc(0x118,0x44,0x3ab,0x47e)+'rgba(75,\x201'+_0x26de55(0x396,0x678,0x300,0x698)+_0x26de55(0x87,-0xa6,0x215,0x3a1)+_0x2af4bc(-0xf5,0x176,0x24,-0x2a9)+'\x20\x20\x20\x20\x20}\x0a\x0a\x20\x20'+_0x26de55(-0xad,0x2a1,-0x63,-0x15e)+_0x2af4bc(-0x15c,0x29a,0xeb,0x30b)+_0x26de55(0xf6,0x3fc,0x362,0x139)+'t::placeho'+_0x2af4bc(-0x10a,-0x399,-0x62,-0x1c4)+_0x26de55(-0xad,0xee,0x1d2,0x1b9)+_0x26de55(-0xad,0x259,-0x2dd,-0x3bb)+_0x26de55(-0xae,-0x111,-0x5a,0x141)+_0x2af4bc(0x58f,0x30d,0x3a3,0x1b7)+_0x2af4bc(0x6e,0x14f,0x24,0x346)+_0x26de55(-0x128,0xa8,-0x41d,0x68)+_0x2af4bc(0x30b,0x29d,0x24,-0x16)+'\x20\x20\x20\x20\x20\x20\x20.bu'+_0x26de55(0x38b,0x1c3,0x3df,0x584)+_0x26de55(-0xad,0x136,0x264,0x243)+_0x2af4bc(-0x28d,0x94,0x24,0x75)+_0x26de55(0x272,0x4e4,-0xc9,0x1c6)+_0x26de55(-0x174,-0x1ee,-0xb7,-0x5a)+_0x2af4bc(-0x2ad,0x2d8,0x24,-0x315)+_0x2af4bc(-0x12b,-0xdd,0x24,-0xe7)+'\x20gap:\x208px;'+'\x0a\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x2d5,0x260,0x24,-0x1ff)+_0x2af4bc(0x266,0x15,-0x6c,-0x15d)+_0x26de55(0x451,0x54e,0x46f,0x2f0)+_0x2af4bc(0x7c8,0x838,0x4eb,0x5ab)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x276,0x7f1,0x598,0x3f6)+_0x2af4bc(-0x17a,0xff,0x24,-0x2d9)+_0x26de55(0x4b6,0x1f6,0x1e1,0x343)+_0x26de55(0x113,-0x181,-0x1d2,-0x22e)+_0x26de55(0x109,-0x24a,0x50,-0x14a)+_0x2af4bc(-0x1a2,0x272,0x24,0x22e)+_0x26de55(-0xad,0x1c0,0xd3,-0x20e)+'position:\x20'+'fixed;\x0a\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x2e2,0x7e,0x24,-0x61)+_0x2af4bc(-0x33,0x119,0x22,-0x4e)+_0x2af4bc(0x3cc,0x4a2,0x1d4,0x469)+_0x2af4bc(0x22c,-0x32d,0x24,0x104)+_0x2af4bc(-0x44,-0x3d,0x2c2,0x257)+_0x26de55(0x26f,0x5c4,-0x2e,0x3c)+_0x2af4bc(-0x1bd,0x4,0x24,0xa0)+_0x2af4bc(-0x206,0x29,0x24,-0x1fa)+_0x26de55(0x3d,-0x328,0x22b,-0x2f9)+_0x26de55(-0x19c,-0x18e,-0x361,-0x4cf)+_0x2af4bc(0x4da,-0xb5,0x24a,0x45)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,-0xbc,-0x50,0x141)+_0x26de55(0x1de,0x3f8,-0xb8,0x95)+'ound:\x20#4CA'+_0x2af4bc(0x5cc,0x53f,0x368,0x589)+_0x26de55(-0xad,-0x1a5,-0x350,-0x17f)+_0x2af4bc(0x554,0x65a,0x481,0x1c1)+_0x2af4bc(-0x309,-0x347,-0x7d,-0x242)+'e;\x0a\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0xa3,0x1f,0x24,0xae)+'\x20\x20\x20\x20\x20\x20\x20pad'+_0x26de55(0x2ac,-0xb,-0x8c,0x368)+_0x2af4bc(0x52c,0x4d3,0x513,0x51c)+_0x26de55(-0xad,0x132,0x8b,0xc0)+_0x26de55(-0xad,0x24b,-0x131,-0x5c)+_0x2af4bc(0x413,0x4f9,0x45e,0x61e)+_0x2af4bc(0x769,0x4e9,0x5bc,0x306)+_0x26de55(0x41a,0x450,0x2c9,0x738)+_0x26de55(-0xad,-0x20c,-0x23b,-0x29a)+'\x20\x20\x20\x20\x20font-'+_0x2af4bc(0x423,0x42d,0x115,-0xa7)+_0x26de55(0x103,0x368,0x21f,0xdf)+_0x2af4bc(0x48,-0x1e5,0x24,-0xf6)+_0x26de55(0x437,0x2f4,0x268,0x324)+'-weight:\x20b'+_0x26de55(0x5c,0xcc,0x247,0x256)+_0x2af4bc(0x5,0x18d,0x24,0x173)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20b'+'ox-shadow:'+'\x200\x204px\x2015p'+'x\x20rgba(0,0'+_0x2af4bc(0x10c,0x65a,0x367,0x4a0)+_0x2af4bc(0xc4,0x5e,0x24,0x369))+(_0x2af4bc(0x181,-0x2f0,0x24,-0x8e)+_0x26de55(0x8e,-0x24f,-0x1bd,-0x1e0)+_0x2af4bc(0x33,0x244,0x118,-0xd4)+_0x2af4bc(0xd9,-0x2e,0x24,-0x203)+_0x2af4bc(-0x336,-0x87,0x24,-0xa2)+'\x20\x20animatio'+_0x2af4bc(0x556,0xfc,0x3be,0x1c3)+'ut\x202s\x20ease'+_0x26de55(0x3f1,0x72f,0x15c,0x3ba)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20}'+_0x2af4bc(-0x2ae,0x84,0x2e,-0x7a)+_0x2af4bc(0x89,0x15c,0x24,-0x288)+_0x26de55(-0x48,0x313,-0xc2,-0x10c)+_0x26de55(-0x1a,-0x36a,-0xd9,0x1d2)+_0x26de55(0x39,-0x108,0x36a,0x1dc)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x200'+_0x26de55(-0xc9,0xe7,-0x209,0x209)+_0x2af4bc(0x4fc,0x386,0x487,0x635)+_0x2af4bc(-0x66,0x210,0x1a9,0x447)+_0x2af4bc(0x11d,0xc9,0xbc,-0x214)+_0x2af4bc(0x2c8,0xca,0x383,0x64f)+_0x2af4bc(-0x70,0x1ab,0x147,0xd2)+_0x26de55(0x1cc,0x15,0x3ed,-0xf5)+_0x2af4bc(0x2f6,0x177,0x24,0x2ed)+_0x26de55(-0x172,-0xf5,-0x40b,-0xe4)+'0%\x20{\x20opaci'+'ty:\x201;\x20tra'+_0x26de55(-0x1d,-0xf7,-0x2fe,0x22)+_0x2af4bc(0x1a2,0x272,0x2e8,0xb7)+_0x2af4bc(-0x2b3,-0x106,-0xad,0x12b)+_0x26de55(0x41e,0x2ba,0x409,0x670)+_0x2af4bc(0x6f,0x343,0x259,0xb9)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x2090%\x20'+_0x2af4bc(0x2a0,0x604,0x318,0x587)+_0x26de55(0x35f,0x2be,0x3,0x686)+_0x2af4bc(0x190,0x569,0x3f9,0x2a3)+_0x26de55(0xb6,0x410,-0x1a,0x12)+_0x2af4bc(0x3e2,-0x39,0x111,0x16c)+'eY(0);\x20}\x0a\x20'+_0x2af4bc(0x257,0x27c,0x24,0x14b)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0x2a,-0x9c,0x179,0x38)+_0x26de55(0x12d,-0x14b,0x467,0x37e)+';\x20transfor'+_0x2af4bc(0x37c,0x223,0x2b8,-0xa2)+_0x26de55(0x31a,0x3f5,-0x3c,0x2f0)+_0x26de55(-0xe1,-0x7e,-0x39,0x156)+_0x26de55(0xce,0x176,0x3fc,0x2af)+_0x26de55(0x41a,0x502,0x197,0x560)+_0x26de55(-0xad,-0x3ee,-0x1a,-0x3ad)+_0x26de55(0x4c7,0x23c,0x502,0x39e)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x163,0x3cc,0x2b3,0x104)+'xt-menu\x20{\x0a'+_0x2af4bc(-0x216,-0x303,0x24,0x11)+_0x26de55(-0xad,0x52,-0x1c3,-0x2fa)+_0x26de55(0x4c,-0x27b,0x167,0x22b)+_0x2af4bc(0x4b3,0x2b9,0x4be,0x267)+_0x2af4bc(0x4b2,0x4ba,0x4eb,0x3be)+_0x2af4bc(0x261,-0xfa,0x24,-0x1b8)+_0x2af4bc(-0x40d,0xd0,-0xb6,-0x327)+_0x26de55(0x462,0x41b,0x58b,0x2cf)+'te;\x0a\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x37b,-0x11f,0x24,-0x1ab)+_0x26de55(0x344,0x1a6,0x3a8,0x1c0)+_0x2af4bc(0x2d0,0x4f9,0x1bd,-0x72)+_0x2af4bc(0x223,-0xbe,-0x13,-0x347)+_0x26de55(0x103,-0x16f,-0x22d,-0x17)+_0x26de55(-0xad,-0x20e,-0x2f8,-0x4c)+'\x20\x20\x20\x20\x20\x20bord'+_0x26de55(0x302,0x4b8,0x3ea,0x547)+'\x208px;\x0a\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x33d,-0x40,0x24,0x1dd)+_0x2af4bc(0x265,0x326,0x429,0x5ff)+_0x2af4bc(0x389,0x2bb,0x11b,0x1e3)+'px\x20rgba(0,'+'0,0,0.2);\x0a'+_0x2af4bc(0x181,0x20c,0x24,0x2a7)+_0x2af4bc(-0x1df,0x19d,0x24,0x354)+_0x26de55(0x32a,0x19f,0x37b,0x3b)+_0x2af4bc(0x100,0xa6,0x3a6,0x678)+_0x2af4bc(-0x2b2,-0x81,0x24,0x2e0)+_0x26de55(-0xad,-0x149,0x1d,-0x295)+'\x20\x20\x20min-wid'+_0x2af4bc(0x19e,0x4bc,0x293,0x177)+_0x26de55(0x41a,0x237,0x4f1,0x249)+_0x26de55(-0xad,-0x27a,-0x3da,-0x7a)+_0x26de55(0xe1,0x3fd,0x220,0x3b3)+_0x2af4bc(-0x1be,-0x1d4,0x16e,0x30b)+_0x26de55(0x41a,0x3d1,0xe2,0x2f2)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x643,0x321,0x598,0x789)+_0x2af4bc(0x9f,0xc1,0x24,-0x24)+_0x26de55(0x163,0x183,-0xcf,0x1f0)+_0x26de55(0x4b1,0x313,0x257,0x6a4)+_0x26de55(-0x1ae,-0x51,-0x65,-0x251))+(_0x2af4bc(0xbd,-0x54,0x24,0x27)+_0x26de55(0x3c5,0x600,0x50f,0x4e6)+_0x2af4bc(0x3b1,0x753,0x446,0x4df)+'px\x2020px;\x0a\x20'+_0x2af4bc(0xe,-0xb8,0x24,0x293)+_0x26de55(-0xad,0x175,0x33,0x258)+_0x2af4bc(0x4fb,0x525,0x500,0x1f6)+_0x26de55(0x2b6,0x39d,0x4f5,0x28a)+_0x2af4bc(-0x84,0x2a,0x24,0x126)+_0x2af4bc(-0x5e,0x342,0x24,0x255)+_0x26de55(-0xf3,-0x3bb,0x172,0x76)+_0x2af4bc(-0x2f6,0x221,-0x31,0xd5)+_0x26de55(-0xad,0x1cc,-0x143,-0x306)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x227,0x2ab,-0x138,0x30b)+':\x2013px;\x0a\x20\x20'+_0x26de55(-0xad,0x3f,-0x33c,0x271)+_0x26de55(-0xad,-0xc2,-0x1e1,-0x126)+_0x2af4bc(0x47c,0x263,0x2c3,0x2bc)+'on:\x20backgr'+_0x26de55(0x424,0x704,0xe4,0x53f)+'\x0a\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,-0x8,-0x173,-0x416)+_0x2af4bc(0x76d,0x27b,0x598,0x724)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20.conte'+'xt-menu-it'+_0x2af4bc(0x1ce,-0x2db,0x5f,0x20f)+_0x2af4bc(0x2a0,0x63c,0x4eb,0x56e)+_0x26de55(-0xad,-0xe3,0x110,-0x242)+_0x2af4bc(0x32,-0x31c,-0xb6,-0xa3)+_0x2af4bc(0x1b0,0xc8,0x2fe,0x238)+_0x26de55(0x70,-0x1ff,-0x36,-0x1e4)+_0x2af4bc(-0x93,0x145,0x24,-0x1b1)+_0x26de55(0x306,0x14d,0x30b,0x148)+_0x26de55(-0xad,-0x10d,-0xa0,-0x1c9)+_0x2af4bc(0x319,0x4da,0x4e9,0x431)+'context-me'+_0x2af4bc(0xf6,-0x41,-0x45,0xa6)+_0x2af4bc(0x551,0x2aa,0x3dc,0x53d)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,-0x225,0x28f,-0x414)+_0x26de55(0x10e,0x299,0x52,0x45a)+'74c3c;\x0a\x20\x20\x20'+_0x2af4bc(-0xb8,-0x22,0x24,0x2a)+_0x26de55(0x13e,0x35f,-0x115,0x358)+_0x26de55(-0xad,-0x2a3,-0xc0,-0x70)+_0x26de55(-0xad,-0x1b2,-0xd7,0xa3)+_0x26de55(0x25d,0xc7,0xe2,0x4d9)+_0x2af4bc(0x3b,-0x373,-0x66,-0x384)+_0x26de55(0x51b,0x7d9,0x339,0x80b)+_0x26de55(0x17e,-0xc8,0xed,-0x134)+_0x2af4bc(0xb7,-0x8e,0x24,-0x266)+'\x20\x20\x20\x20\x20\x20\x20\x20ba'+'ckground:\x20'+_0x2af4bc(-0x113,-0x117,0x1d1,0x268)+_0x2af4bc(-0x1bd,0x2bb,0x24,-0x1cf)+_0x2af4bc(0x19b,-0x1b,0x25f,0x175)+_0x2af4bc(0x339,-0x170,0x24,-0x2dd)+_0x2af4bc(0x6b9,0x3e1,0x50b,0x733)+_0x26de55(0x3e5,0x409,0x543,0x393)+_0x2af4bc(-0x3f4,0x2a3,-0x9b,0x56)+'ame\x22],\x0a\x20\x20\x20'+_0x2af4bc(0x1e7,-0x233,0x24,-0x83)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20td:nth-ch'+_0x2af4bc(-0x1fb,0x26f,0x4e,-0x246)+_0x26de55(-0xad,0x186,-0x342,-0x1cc)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(-0x98,-0x4d,0x4c,-0x4d)+'th:\x2080px;\x0a'+_0x26de55(-0xad,0x4d,-0x172,-0x201)+_0x26de55(-0xad,-0x3c9,-0x1d0,-0x379)+'\x20\x20\x20\x20\x20\x20\x20\x20ma'+_0x2af4bc(0x493,0x60c,0x3e5,0x111)+'0px;\x0a\x20\x20\x20\x20\x20'+_0x2af4bc(-0x2,0x20d,0x24,-0x1dc)+_0x2af4bc(0x3fa,0x5dd,0x4a9,0x421)+'\x0a\x20\x20\x20\x20\x20\x20\x20\x20<'+_0x2af4bc(0x3d9,0x4c9,0x214,0x518)+_0x26de55(0x232,-0x104,-0x8,-0x107)+_0x2af4bc(0xbc,0x21b,0x1c3,0x3af)+'>\x0a\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x6ae,0x24e,0x41b,0x3d2)+_0x26de55(-0xc7,0x1c4,-0xea,-0x1c1)+'r\x22>\x0a\x20\x20\x20\x20\x20\x20'+_0x26de55(0x1a7,0x58,-0xa3,0x19f)+'Non-Destru'+_0x2af4bc(0x284,0x4db,0x3ff,0x755)+_0x26de55(0x294,0x151,0x444,0x32f)+'ditor</h1>'+_0x2af4bc(0x4d7,0x63d,0x4eb,0x34f)+_0x2af4bc(0x1f5,-0x262,0x5e,-0x158)+_0x26de55(0x384,0x34,0x697,0x369)+_0x26de55(-0xb7,0x12f,-0x17c,-0x258)+_0x2af4bc(0x47e,0x4d1,0x209,0x377)+_0x26de55(0x63,-0x2d5,0x26c,0xff)+_0x26de55(0x282,0x4df,0x50a,0x58a)+_0x2af4bc(0x2bb,-0xfe,0x124,0x36c)+_0x2af4bc(0xbc,0x3c1,0x1a7,0x4cc))+('om:\x2010px;\x20'+'margin-top'+_0x2af4bc(0x810,0x5c4,0x5aa,0x696)+_0x2af4bc(0x368,-0x282,0x24,0x374)+_0x26de55(0x2b3,0x1af,-0x25,0x493)+'25\x20by\x20Sang'+_0x26de55(0x443,0x631,0x16c,0x66a)+'\x0a\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x31c,0x5a9,0x380,0x33e)+_0x2af4bc(0x162,0x3ba,0x336,0x5b3)+_0x2af4bc(0x64,0x14a,0x17a,0x486)+_0x26de55(0x2e4,0x106,0x20b,0x458)+_0x2af4bc(-0x222,0x1c4,0x13d,0x334)+_0x26de55(0x48d,0x5bc,0x241,0x195)+_0x2af4bc(0x41c,0x444,0x43e,0x395)+_0x2af4bc(0x4df,0x128,0x26a,0x367)+_0x2af4bc(-0x2a,-0x199,-0x82,0x1e1)+_0x2af4bc(0x109,0x418,0x1a6,0x332)+_0x2af4bc(-0x19,0x2b9,0x8c,0x2f4)+_0x2af4bc(0x456,0x533,0x35e,0x4af)+_0x26de55(0x2e0,-0x89,0xea,0x21c)+'>\x0a\x0a\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x12b,0x199,0x2c,-0x26c)+_0x26de55(-0xc0,0x266,-0x84,-0x222)+_0x2af4bc(0x90,0x11a,0x2ab,0x445)+_0x2af4bc(0x17a,0x54c,0x442,0x456)+_0x2af4bc(0x428,0x1af,0x37f,0x27b)+_0x26de55(0x1ac,0x441,-0x191,0x1c6)+_0x2af4bc(0x268,0x31a,0x5b5,0x2d8)+_0x2af4bc(-0x219,-0x169,0x119,0x318)+'de\x20them\x20|\x20'+_0x2af4bc(-0x268,-0x1fd,0x6a,-0x2c6)+_0x2af4bc(0x3d0,0x3cf,0x4b9,0x4e8)+_0x26de55(-0x11d,-0x6b,-0x1f2,-0x25a)+_0x26de55(-0xa,-0x20c,0x347,0x2e9)+_0x2af4bc(-0x90,-0x27e,0xd0,0x233)+_0x2af4bc(-0x95,0x285,0x98,0x2fe)+_0x2af4bc(0x2af,0x80c,0x516,0x20b)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x235,-0x2c,-0x51,0x590)+_0x26de55(0x27f,0x56a,0x86,0x3f2)+_0x2af4bc(-0x1c4,0x38,-0x44,-0x36b)+_0x2af4bc(0x4a,0x263,0x15a,0x2c2)+_0x26de55(-0x163,-0x187,0xf2,-0x188)+_0x26de55(0x4fc,0x40d,0x236,0x1e8)+'our\x20whole\x20'+_0x26de55(0x11b,0x42d,0x6a,0x2c9)+_0x2af4bc(0x245,-0x8b,0x23d,0x4c3)+'or\x20transla'+_0x2af4bc(-0xf,0x2cb,0x12,-0x2e9)+'ave\x20&\x20Appl'+_0x2af4bc(-0x283,-0xed,0xd4,0x2b6)+_0x26de55(-0x147,0x203,-0x4c,-0x1af)+_0x2af4bc(0x5b7,0x6bc,0x44a,0x24c)+_0x2af4bc(0x378,0x662,0x544,0x808)+_0x2af4bc(0x274,0x20,0x52,0x77)+_0x2af4bc(0xe7,0x39d,0x6c,-0x2b0)+_0x26de55(0x13,0x25,0x160,-0x225)+_0x2af4bc(-0x4a,0x324,0x74,0x354)+'op-control'+_0x26de55(-0x1ad,-0x30d,-0x229,-0xce)+_0x2af4bc(-0x2bf,0x340,0x24,-0x57)+_0x26de55(0x34a,0x251,0x470,0x292)+_0x26de55(0x185,-0x195,0x3df,-0x13a)+_0x2af4bc(0xca,0x4a2,0x27b,0x3d)+_0x26de55(0x41a,0x44f,0x272,0x210)+_0x26de55(-0xad,-0x26a,-0xbc,-0x1ca)+_0x26de55(-0x7e,0x1a3,-0x228,0x2b1)+'pe=\x22text\x22\x20'+_0x2af4bc(0x2a7,0x208,0x4c3,0x1c5)+_0x26de55(0x1b4,0x355,0x3c3,0x455)+_0x26de55(0x127,0x3c8,-0xfc,0x31f)+'-input\x22\x20pl'+_0x26de55(0x172,0x1dd,0x426,0x276)+_0x26de55(0x417,0x1f9,0x4de,0x335)+'text...\x22\x20/'+_0x2af4bc(-0x161,0x152,0x52,0x20)+'\x20\x20\x20\x20\x20\x20\x20\x20</'+'div>\x0a\x20\x20\x20\x20\x20'+_0x2af4bc(0x26f,-0xb,0x24,0x1f0)+_0x2af4bc(0x1c1,0x3e,0x250,0x1d5)+_0x26de55(-0x93,-0x9,-0x2d6,-0x263)+_0x26de55(0x43,-0x25f,-0x1e5,0xf1)+_0x26de55(-0xad,0x195,-0x127,0x21e)+_0x2af4bc(0x313,0x394,0x2b4,0x529)+_0x2af4bc(0x704,0x265,0x590,0x78b)+'extractCSV'+_0x2af4bc(-0x1e0,-0x31f,-0x16,0x6c)+'act\x20CSV</b'+_0x2af4bc(-0x28c,0x6f,0x7c,-0x8d)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20<bu'+_0x26de55(0x3b1,0xd5,0x6d5,0x41c)+_0x26de55(0xc0,0x30e,-0x227,0x83)+_0x2af4bc(0x4f3,-0x1e,0x1f7,0x2c6)+_0x26de55(0x1d1,0x33c,0x13f,0x20c)+_0x2af4bc(0x440,0x5a8,0x4ce,0x48d)+_0x2af4bc(0x51e,0xe4,0x2e0,0x45c)+'\x0a\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(-0x32c,-0x162,0x24,-0x325))+('\x20<button\x20o'+'nclick=\x22sa'+_0x26de55(0x3f5,0x599,0x5ea,0x270)+'\x20Save\x20&\x20Ap'+_0x26de55(-0x149,-0x2dd,-0x491,-0x320)+'n>\x0a\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x35a,-0xc9,0x24,-0x97)+_0x2af4bc(0x23e,0x4b1,0x2b4,0x47d)+_0x2af4bc(0x8af,0x539,0x590,0x812)+_0x26de55(0x19a,0x40d,0x2cc,-0xe7)+_0x26de55(0x3e7,0x72a,0x6db,0x39d)+_0x26de55(0xe6,0x1cd,0x2da,0x32f)+'ton>\x0a\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xb1,-0x2e5,-0x33b,0x1d9)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0xd2,0x435,-0x130,-0x12a)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(-0x4f,0x24d,0x250,0x44a)+_0x26de55(0x391,0x4ed,0x688,0x5da)+'ontainer\x22\x20'+_0x2af4bc(0x487,0x3af,0x12b,0x471)+'ontainer\x22>'+'\x0a\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20<ta'+'ble\x20id=\x22cs'+_0x2af4bc(0x2c9,0x1ab,0x426,0xdb)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x324,0x2fc,0x2c1,0xf9)+_0x2af4bc(-0x13,0x315,0x1b3,0x2)+_0x26de55(0x281,0x2ff,0x56e,0x10b)+_0x2af4bc(0x73,0x156,0x18a,0x179)+_0x2af4bc(-0xbb,-0x74,0x52,-0x13c)+_0x2af4bc(-0x1c6,-0x21,0x24,0x19c)+_0x2af4bc(0x1bf,-0x37,-0xc8,0x173)+_0x2af4bc(0x61a,0x4de,0x4a8,0x328)+_0x26de55(0xcd,0x2b8,-0x42,-0xae)+_0x2af4bc(0x5f1,0x37b,0x346,0x411)+_0x26de55(0x324,0x585,0x217,0x5ad)+_0x26de55(0x431,0x278,0x660,0x194)+_0x26de55(-0xad,0x39,-0x216,-0x2de)+_0x2af4bc(0x420,0x54d,0x5f0,0x409)+_0x2af4bc(0x33c,-0x54,0x37,-0xbb)+_0x26de55(0x7f,0x2bc,0x36e,-0x7)+_0x26de55(0x427,0xe4,0x50c,0x5eb)+_0x2af4bc(-0x2d,-0x1af,-0xbd,-0x417)+'nction\x20sho'+_0x2af4bc(0x422,0x489,0x563,0x790)+'Message(me'+'ssage)\x20{\x0a\x20'+_0x26de55(-0xad,-0x29a,-0xbd,0x236)+'\x20const\x20exi'+_0x2af4bc(-0x2e,0x129,-0xcf,-0x294)+_0x26de55(0x18d,0x13,-0xa4,-0xed)+_0x2af4bc(0x28e,0x212,0x483,0x43b)+_0x2af4bc(0x524,0x1c8,0x418,0x575)+_0x2af4bc(0x14b,0x211,0x295,0x12d)+_0x2af4bc(-0x67,-0x148,0x159,0x3c8)+_0x26de55(-0xad,-0x29,-0x1a2,0x10)+'if\x20(existi'+'ngMsg)\x20{\x0a\x20'+_0x2af4bc(0x11d,0x2ae,0x24,0x1dc)+_0x26de55(0x20b,0x219,0x2a8,-0xc9)+'ingMsg.rem'+_0x2af4bc(0x502,0x783,0x4f1,0x32e)+_0x2af4bc(0x2c7,0x15b,0x4a9,0x1ee)+_0x2af4bc(0x1d2,0x31e,0x4eb,0x424)+_0x2af4bc(0x70c,0x23e,0x3c5,0x150)+_0x26de55(-0x8f,0x10d,0x4a,-0x2e6)+_0x26de55(-0x166,-0x3d0,-0x2b5,-0x26c)+_0x2af4bc(0x291,-0xc6,0x279,0x5d0)+_0x2af4bc(0x49c,-0x38,0x17d,0x148)+'Element(\x27d'+_0x26de55(-0x59,0x159,0x1a0,0x35)+_0x26de55(-0x8d,0x3,-0x33,0x162)+_0x2af4bc(0x55c,0x4e2,0x2d3,0xb7)+'lassName\x20='+'\x20\x27auto-clo'+_0x26de55(0x1dd,0x253,0x2ec,0x404)+_0x26de55(0x4e2,0x1ae,0x1d1,0x346)+_0x2af4bc(-0x14d,0x37a,0xec,0x1f3)+_0x2af4bc(0x28b,0x4cb,0x2c6,0x3be)+_0x26de55(0x31c,0x21e,0x3da,0xc3)+_0x26de55(0x4c5,0x47c,0x1eb,0x5b9)+_0x2af4bc(0x246,0x12f,0x24,0x39)+_0x2af4bc(0x81d,0x5a3,0x5e7,0x3fc)+'\x20\x20\x20\x20docume'+_0x26de55(-0x138,0x13b,0xbc,-0x3e)+_0x26de55(0x117,0x276,-0x1a5,0xb9)+_0x2af4bc(0x5be,0x6c1,0x405,0xf8)+_0x2af4bc(0x47a,0x44d,0x48f,0x440)+_0x2af4bc(0x5f7,0x2c3,0x438,0x20b)+_0x2af4bc(0x22c,0x39f,0x342,0x2da)+'tTimeout(('+')\x20=>\x20{\x0a\x20\x20\x20'+_0x2af4bc(-0x18d,-0x318,0x24,-0x135)+'\x20\x20\x20if\x20(mes'+_0x26de55(0x477,0x3c3,0x2d3,0x78a)+_0x26de55(0x8b,0x29c,-0xd7,-0x81)+_0x26de55(-0x129,0x228,-0x1dc,-0x373))+(_0x26de55(0x41b,0xde,0x568,0x590)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x1e1,0x2e9,-0xb1,-0x154)+'ageDiv.rem'+'ove();\x0a\x20\x20\x20'+_0x26de55(-0xad,-0x194,-0x34b,-0x71)+_0x26de55(-0x77,-0x20d,0x151,0xeb)+_0x26de55(0x242,0x4a4,0xf3,-0x109)+_0x26de55(0x357,0x5e2,0x3a5,0x10)+_0x2af4bc(0x3bd,-0x88,0x181,0x319)+_0x26de55(0x345,0x59e,0x359,0x44d)+_0x26de55(0x1d6,0x265,0x1c5,-0x158)+_0x2af4bc(0x4ac,0x219,0x29c,-0x5b)+'et();\x0a\x0avar'+_0x26de55(0xcf,0x278,-0x22d,0x2c1)+_0x26de55(0x1fb,0x410,0x288,-0x4b)+'null;\x0a\x0afun'+_0x2af4bc(0x39f,-0x1aa,0x3d,-0x242)+_0x2af4bc(-0x2f9,-0x40,0x2d,-0x91)+_0x2af4bc(0x12f,0x170,0x3ec,0x186)+_0x26de55(0xa5,0x2de,0x285,0x11a)+'\x20\x20\x20e.preve'+_0x26de55(0x4ae,0x7ff,0x7a9,0x353)+_0x2af4bc(0x359,0x135,0x81,-0x293)+_0x26de55(0x42c,0x220,0x66a,0x177)+'tingMenu\x20='+_0x2af4bc(-0xef,0x33c,0x25e,0x230)+_0x26de55(0x3b2,0x6da,0x57f,0x9e)+_0x2af4bc(0x50e,0x5f8,0x4c0,0x662)+_0x26de55(0x27,-0x10e,0xf0,-0x17b)+_0x26de55(0x472,0x308,0x168,0x37e)+_0x2af4bc(0x52a,0x7c0,0x5a0,0x85c)+_0x26de55(0x25c,0x575,0x466,0x1be)+'\x20\x20\x20\x20existi'+_0x2af4bc(0x27a,0x507,0x4ed,0x583)+_0x2af4bc(0x5c5,0x858,0x4f1,0x7a5)+_0x2af4bc(-0x3d,-0x1e6,0x13c,-0x74)+_0x26de55(0x249,0x184,0xaa,-0x38)+_0x26de55(0x39b,0x6c,0x55b,0x52a)+_0x26de55(0x4ed,0x5fb,0x62d,0x3a9)+_0x2af4bc(0x104,-0x354,-0x4e,-0x1df)+_0x2af4bc(0x41d,0x349,0x59c,0x457)+'nu.classNa'+_0x26de55(0x4ee,0x4b9,0x2e4,0x24c)+_0x2af4bc(0x6c9,0x24a,0x472,0x6fb)+_0x26de55(0x36f,0xff,0x217,0x3e)+'style.left'+'\x20=\x20e.clien'+'tX\x20+\x20\x27px\x27;'+_0x2af4bc(0x367,0x23f,0x440,0x4ca)+_0x2af4bc(0x72b,0x496,0x419,0x736)+'=\x20e.client'+_0x2af4bc(0xb2,0x31e,0x351,0x1e7)+_0x26de55(0x36a,0x24f,0x427,0x173)+_0x2af4bc(0x34f,0x3d1,0x316,0x18)+_0x26de55(0x18c,0x16e,0x4df,-0x151)+_0x2af4bc(0x208,0x1ad,0x4e7,0x6db)+_0x2af4bc(0x354,0x5b4,0x5cf,0x63e)+_0x26de55(0x7d,-0x130,0x61,0x2a4)+_0x2af4bc(0x318,0x259,0x1f2,0x45d)+'n.classNam'+_0x2af4bc(0x2a1,0xa0,0x136,0x2a8)+_0x26de55(0x4b1,0x3dd,0x520,0x6fc)+_0x2af4bc(0x2d6,0x24d,0x4e6,0x747)+_0x2af4bc(0x459,0x105,0x162,0x37a)+_0x2af4bc(0x173,0x8a,0x283,-0xe2)+_0x26de55(-0x52,0x1fe,-0x2e5,-0x1cb)+_0x2af4bc(0x7e,-0x9c,0x35,0x1a0)+_0x2af4bc(0x1ce,-0x1d0,-0x51,0x1c1)+_0x2af4bc(0x2fe,0x51c,0x1f2,-0x13a)+_0x26de55(0x4f9,0x44a,0x43c,0x35e)+_0x2af4bc(0x177,-0xa1,0xe2,0x1e2)+_0x26de55(0x54,0x335,-0x28,0x39f)+_0x26de55(0x189,0x20f,-0xbe,0x4a)+_0x26de55(0x3fe,0x4ca,0x5d2,0x6f3)+_0x26de55(0x25a,0x42f,0x4db,0x2d3)+';\x0a\x20\x20\x20\x20\x20\x20\x20\x20'+'menu.remov'+_0x26de55(0x128,0x1ef,0x1e3,0x36d)+';\x0a\x20\x20\x20\x20\x0a\x20\x20\x20'+_0x26de55(0x4b4,0x241,0x5ef,0x7d6)+_0x2af4bc(0x2dc,0x143,-0x17,0x190)+_0x2af4bc(0x394,0x351,0x28b,0x90)+_0x2af4bc(-0x1ac,0xc6,-0x81,-0x1ff)+'t(\x27div\x27);\x0a'+_0x26de55(0x158,0x74,0xe,0x470)+_0x26de55(0xcb,-0x219,-0x1a2,0x6e)+_0x26de55(-0xd5,0xc4,-0xd5,-0x25a)+_0x2af4bc(0x345,-0x135,0x213,-0x9e)+_0x2af4bc(0x138,0x4a1,0x3fe,0x50f)+'\x20\x20\x20pasteOp'+_0x2af4bc(0x2d8,0x401,0xc1,0x228)+'ontent\x20=\x20\x27'+'📥\x20Paste\x20To'+'\x20This\x20Colu'+'mn\x27;\x0a\x20\x20\x20\x20i'+_0x2af4bc(0x1f0,0x523,0x530,0x896)+_0x26de55(0x248,0x20f,0x2f1,0x58d)+')\x20{\x0a\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x331,0x3ba,0x5d,0x24b))+(_0x26de55(0x2d3,0x60a,0x608,0x331)+_0x26de55(0x4ce,0x7d2,0x699,0x1fd)+_0x2af4bc(0x6a1,0x384,0x55f,0x603)+_0x26de55(-0x1ab,-0x3d7,0x14f,-0x31f)+'Option.sty'+_0x26de55(0x463,0x47d,0x295,0x4d3)+'=\x20\x27not-all'+_0x2af4bc(0x44c,0x452,0x172,0x103)+_0x26de55(0x512,0x6c4,0x704,0x6e6)+_0x26de55(0x10b,-0x1e4,0xad,0x44f)+'steOption.'+_0x2af4bc(0x2a1,0x329,0x551,0x3ea)+_0x2af4bc(-0xc9,0x24c,-0x8d,-0x30e)+_0x26de55(0x3bf,0xe3,0x33b,0x6ca)+'\x20\x20\x20\x20\x20paste'+_0x26de55(-0xa0,-0x2d7,-0x1af,0x1d4)+_0x2af4bc(0x47c,0x1cb,0x32b,0x42)+';\x0a\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20menu.r'+_0x2af4bc(-0xcb,0x3cb,0x8a,-0x18)+_0x26de55(0x34f,0x5e2,0x533,0x1d7)+'\x20\x20\x20\x20}\x0a\x20\x20\x20\x20'+_0x2af4bc(0x2b2,-0x87,0x1b8,0x32e)+_0x2af4bc(-0x1b3,0x2e8,-0x14,0x346)+_0x26de55(0x18d,0x104,0x45b,-0x1d8)+_0x2af4bc(0x2c1,0x163,0x32c,0x3bd)+_0x26de55(0x14b,0x2cf,0x356,0x2f1)+_0x26de55(-0xdc,0x7c,-0x368,-0x47)+_0x2af4bc(-0x174,0x2c2,0xd3,-0x3d)+_0x26de55(0x381,0x545,0x68c,0x248)+_0x26de55(0x4b,0x1c0,-0x22d,0x3c)+_0x26de55(0x4cc,0x316,0x24d,0x733)+';\x0a\x20\x20\x20\x20sepa'+_0x26de55(0x2,0x17a,-0x223,0x271)+_0x2af4bc(0x23a,0x382,0x1ce,-0x100)+_0x2af4bc(0x108,-0x204,0xa9,0xcf)+'\x20\x20\x20\x20\x0a\x20\x20\x20\x20v'+_0x26de55(0x3a,-0x74,0x83,-0x17e)+_0x26de55(0x18c,0x4df,-0x7,0x226)+'ment.creat'+'eElement(\x27'+_0x2af4bc(-0xe6,0x1a7,0x14e,0x331)+_0x26de55(0x1c7,0x40b,0x19b,0xe6)+_0x26de55(0x32,0xd9,-0x1bf,-0x20b)+'e\x20=\x20\x27conte'+_0x26de55(0x4b1,0x5bd,0x749,0x39e)+_0x2af4bc(0x70,-0x188,0x4a,-0x110)+_0x26de55(0x36,-0x149,0x143,-0x84)+_0x26de55(0x1e5,0x39e,0x3f1,-0x72)+_0x26de55(0x4ca,0x4d6,0x7ef,0x226)+_0x26de55(0x5e,-0x180,-0x1a4,0xe2)+_0x2af4bc(0x770,0x34a,0x56e,0x7b5)+'ace\x27;\x0a\x20\x20\x20\x20'+_0x26de55(0x312,0x403,0x555,0x434)+_0x2af4bc(-0x185,0x434,0x199,-0x123)+_0x2af4bc(0xe5,0x355,0x22f,-0x78)+')\x20{\x0a\x20\x20\x20\x20\x20\x20'+'\x20\x20hideColu'+_0x26de55(0x30a,0x7,0x352,0x3bd)+_0x2af4bc(0x131,0x544,0x329,0x47b)+_0x2af4bc(0x582,0x4cd,0x507,0x422)+_0x26de55(-0x17f,0x152,-0x300,0x165)+_0x2af4bc(0x2f4,0x51b,0x373,0x1a)+_0x2af4bc(0x4ac,0x339,0x33d,0xc3)+'\x20\x20\x20\x0a\x20\x20\x20\x20me'+_0x2af4bc(0x180,-0x2a0,0xb,-0xd)+'hild(copyO'+_0x26de55(0xd9,-0x3f,0x38f,0xf4)+_0x2af4bc(0x735,0x1a2,0x432,0x639)+_0x26de55(0x16,-0x185,-0xee,-0x1bd)+_0x26de55(0x147,0x33d,0x35b,0x36e)+_0x2af4bc(0x496,0x5e3,0x583,0x48d)+_0x2af4bc(-0x4e,-0x194,-0x77,-0x177)+'ild(separa'+_0x2af4bc(0x26f,0x13f,0x2a3,0x2b1)+'menu.appen'+_0x26de55(0x1e4,-0x15f,0xdf,0x305)+'eOption);\x0a'+_0x2af4bc(0x554,0x5bb,0x4f9,0x418)+_0x26de55(-0x138,-0x254,-0x1e3,0x21)+_0x26de55(0x117,0x31,-0x22e,0x200)+'menu);\x0a\x20\x20\x20'+'\x20\x0a\x20\x20\x20\x20setT'+'imeout(fun'+_0x2af4bc(0x294,-0x3f,0x95,-0x176)+_0x2af4bc(0x16e,0x25a,0x333,0x159)+_0x2af4bc(0xab,0x353,0x23f,0x501)+_0x2af4bc(0x3cf,-0x14a,0xba,0x10f)+_0x2af4bc(0x3a3,0x12a,0x13b,-0x45)+_0x2af4bc(-0x11d,-0x385,-0xce,-0x3fc)+_0x26de55(0xe4,0x2cb,0x339,-0x1b1)+_0x26de55(0x23d,0x3f0,0x30b,0x19)+_0x26de55(-0x8d,-0x18a,-0x166,0x2c1)+_0x2af4bc(0x1ef,0x495,0x3a5,0x158)+');\x0a\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x5a,-0x44,0x18d,0x103)+_0x26de55(0x207,0x3d0,0x51f,0xd9)+_0x26de55(-0x17,-0xfd,-0x1af,-0x25c)+'ner(\x27click'+_0x26de55(0x1a0,0x83,0x2ed,0x2da))+('nu);\x0a\x20\x20\x20\x20\x20'+_0x2af4bc(0x4b3,0x488,0x23a,0x70)+_0x26de55(0x2c5,-0x96,0x24e,0x599)+'}\x0a\x0afunctio'+'n\x20copyWhol'+_0x2af4bc(0x2da,0x9f,0x33,0x100)+_0x2af4bc(0x6fc,0x547,0x423,0x237)+_0x26de55(0x119,0x36a,0x332,0x17)+'header\x20=\x20d'+_0x2af4bc(0x2f3,0x6f7,0x3aa,0x66e)+'tElementBy'+_0x26de55(-0x8,0x327,-0x24a,0x46)+_0x2af4bc(0x5,0x152,0x34,0x2c9)+_0x2af4bc(0x49e,0x47d,0x570,0x3bd)+_0x26de55(0x12a,-0x69,0x11,0x416)+'nt.getElem'+_0x2af4bc(0x4eb,0x121,0x237,0x58f)+'ableBody\x27)'+_0x2af4bc(0x361,0x47,0x305,-0x59)+_0x26de55(0x410,0x1e4,0x329,0x42c)+_0x2af4bc(0x750,0x7dc,0x52b,0x5b0)+_0x26de55(0x37e,0x2d,0x254,0x40d)+_0x2af4bc(0x5b8,0x470,0x3c7,0x23f)+_0x2af4bc(-0xf9,0x1c6,0xd6,-0x220)+_0x2af4bc(0x223,0x643,0x4df,0x463)+'Index\x20=\x20-1'+_0x2af4bc(-0x50,0x151,0x1c9,0x30f)+'ers.forEac'+'h(function'+'(th,\x20index'+_0x2af4bc(0x3fc,0x4e2,0x486,0x173)+_0x26de55(-0x191,-0x3ed,-0x3fd,0x3a)+'ataset.col'+'umnName\x20=='+_0x2af4bc(0x7d6,0x7ad,0x5e8,0x530)+_0x26de55(-0x11,-0x1d0,-0x1e0,-0x212)+_0x2af4bc(0x2a0,0x370,0x2be,0x33f)+_0x2af4bc(-0x2f,0x1fd,0xff,0x414)+_0x2af4bc(-0x36,0x1c1,-0x90,-0x45)+'\x20\x20\x20\x20\x20\x20\x20}\x0a\x20'+'\x20\x20\x20});\x0a\x20\x20\x20'+_0x26de55(0xf9,-0x5,0x12,0x1a0)+_0x2af4bc(0x128,0x1b4,0x28f,0x310)+_0x2af4bc(0x4f,-0xd8,-0x43,0x135)+_0x2af4bc(-0xdf,0x27f,0x9b,0x12f)+_0x2af4bc(-0xb8,-0x3f,0x1f4,0x41d)+_0x26de55(0x6f,0x3b,0x285,-0x28b)+_0x2af4bc(0x9d,-0x1b,-0x69,-0xac)+_0x26de55(0x12b,-0x76,0x369,-0x28)+_0x26de55(0x110,-0x80,-0x1f6,-0xa4)+_0x2af4bc(0x242,0x243,0x483,0x2ba)+'torAll(\x27tr'+'\x27);\x0a\x20\x20\x20\x20ro'+'ws.forEach'+_0x26de55(0x34e,0xfc,0x3fa,0x4a7)+_0x2af4bc(-0xf,0xf2,0x2b,0x2e6)+_0x26de55(0x11d,0x5d,0x373,0x1e8)+'ell\x20=\x20row.'+_0x26de55(0x4f4,0x274,0x575,0x851)+_0x2af4bc(0x9d,0x152,0x3fa,0x4f5)+_0x26de55(0x4c9,0x2ec,0x4ed,0x7c0)+_0x26de55(0x10c,0x54,-0xc,-0x86)+_0x2af4bc(0x1b8,0x5cd,0x490,0x68a)+_0x26de55(0x21c,-0x8d,-0xe9,0x3a9)+_0x2af4bc(-0x61,0x4f4,0x18e,0x282)+_0x2af4bc(0xcb,0x1d9,0xee,0x134)+_0x2af4bc(0x2f,0x52f,0x38c,0x1db)+_0x26de55(0x337,0x288,0x2c,0x2a4)+';\x0a\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x328,0x39f,0x3bb,0x107)+_0x2af4bc(0x3dd,0xb5,0xe8,0xc)+_0x2af4bc(0x23b,0x1cc,0x24,0x209)+_0x26de55(-0x183,-0x12a,0xe2,-0x2d5)+_0x2af4bc(0x625,0x392,0x4c1,0x697)+_0x2af4bc(0x56d,0x392,0x2ff,0x163)+_0x2af4bc(0x98,-0x1b4,-0xa0,-0x400)+_0x2af4bc(0x712,0x187,0x46d,0x265)+_0x2af4bc(0x17e,0x8a,0x202,0x4bf)+_0x26de55(0x131,0x2df,0x56,0x494)+_0x2af4bc(0xb,0x131,0x23a,0x177)+_0x26de55(0x11f,0x37,0x22a,-0x163)+_0x26de55(0x414,0x6c5,0x117,0x5ca)+'essage(\x27Co'+_0x2af4bc(0x83,-0xa4,-0xb0,-0x314)+_0x2af4bc(0x19c,0x30e,0x370,0x1ce)+_0x2af4bc(-0x5d,-0x21e,-0x7,0x1d0)+'\x27\x22\x27);\x0a}\x0afu'+_0x2af4bc(-0x2c6,-0x294,-0x9f,-0x381)+_0x26de55(0x40a,0x326,0x589,0x55e)+_0x2af4bc(0x3fb,0x5ab,0x2d6,0x31)+_0x26de55(-0x193,-0x48b,-0x139,-0x169)+_0x2af4bc(0x301,0x63c,0x530,0x424)+_0x26de55(0x248,0x320,0x4f,-0x37)+'\x20||\x20copied'+_0x26de55(0x248,0x558,0xfb,-0x98)+'.length\x20=='+_0x26de55(0x73,0x231,0x232,-0xb6)+_0x26de55(-0x12d,0x1f9,-0x2b,-0x297)+_0x2af4bc(-0xd5,0x6a,0xaa,0xe3)+_0x26de55(0xdb,0x31,0x12b,-0x19b))+(_0x2af4bc(0x45f,0x33b,0x1ae,-0x1a3)+_0x2af4bc(0x290,0x3,0x25b,0x3ea)+_0x2af4bc(0x101,-0x91,0x241,0x333)+'\x20var\x20messa'+_0x2af4bc(-0x24f,0x1a,0xfe,0x2f6)+'e\x20to\x20this\x20'+_0x2af4bc(0x630,0x234,0x557,0x51e)+_0x26de55(-0xab,-0x281,0x244,-0x114)+_0x26de55(-0x1a7,0x116,-0x292,-0x99)+_0x26de55(0x229,0x1df,0x446,0x2b7)+_0x2af4bc(0x1d2,0x1a3,-0x7f,0x1ae)+_0x26de55(0xfc,0x406,0x440,0x2b2)+_0x2af4bc(0x4e,-0x2a9,-0xb3,0x78)+_0x26de55(0x2c4,0x240,0x5ba,0x414)+_0x26de55(0x425,0x435,0x589,0x4b6)+_0x26de55(0x4b5,0x369,0x26a,0x3da)+_0x26de55(0x228,0x28e,0x56c,0x493)+'rn;\x0a\x20\x20\x20\x20\x0a\x20'+_0x26de55(-0xb5,-0x1b4,0x27a,-0x76)+_0x2af4bc(0x305,0x683,0x57d,0x405)+_0x2af4bc(0x200,0x2cd,0x2a0,0x5ae)+_0x26de55(-0x13f,0x94,0x85,0x186)+'\x27tableHead'+_0x2af4bc(0x3dd,0x4ca,0x537,0x237)+_0x2af4bc(0x461,0x6ce,0x4cb,0x30e)+_0x2af4bc(0x5a0,-0xf5,0x25e,0x122)+_0x2af4bc(0x2fa,-0x1f0,0x135,0xbf)+_0x2af4bc(0x350,0x24d,0x573,0x3c6)+_0x26de55(0x3f9,0x2b4,0x232,0x249)+_0x2af4bc(0x13a,0x31a,0x304,0x27e)+'r\x20headers\x20'+_0x26de55(0x9c,0x2d7,-0x140,0x31f)+_0x26de55(0x310,0x41e,0x4f2,0x22c)+_0x2af4bc(0x53b,0x1ae,0x301,0x490)+_0x26de55(-0x2f,-0x50,0x257,0x2bc)+_0x2af4bc(0x3c0,0x7e2,0x4f0,0x83c)+_0x2af4bc(0x3b0,-0x1e,0x4f,0x33b)+_0x26de55(0x3d5,0x3aa,0x1d8,0x707)+_0x2af4bc(-0x3f7,-0x4b,-0xdb,-0x2b0)+'unction(th'+_0x2af4bc(0x6c5,0x6b9,0x5d4,0x3e9)+_0x26de55(0x447,0x30a,0x339,0x1a8)+_0x2af4bc(0x3a6,0x373,0x1c0,0x2bf)+_0x26de55(-0x63,-0x22e,-0x96,0xe)+_0x26de55(0x38c,0x6d2,0x159,0x5d2)+'olumnName)'+_0x26de55(0x3bf,0x30e,0x25f,0x98)+'\x20\x20\x20\x20\x20colum'+_0x26de55(0x4b7,0x608,0x68d,0x31f)+_0x2af4bc(-0x89,-0x13a,0x70,-0x247)+_0x2af4bc(0x389,0x2c1,0x3e4,0x99)+'});\x0a\x20\x20\x20\x20\x0a\x20'+'\x20\x20\x20if\x20(col'+_0x26de55(-0x26,-0x22,0xd9,-0x266)+'==\x20-1)\x20ret'+_0x26de55(0xee,0x308,0x162,-0x165)+_0x2af4bc(-0x97,-0x20,0x1fc,0x243)+_0x26de55(0x110,-0x1bb,0x354,-0x60)+_0x26de55(0x3b2,0x488,0xeb,0x425)+_0x2af4bc(0x384,0x3ba,0x29a,0x361)+_0x26de55(-0x103,-0x257,-0x331,0x1dd)+'r\x20pastedCo'+_0x2af4bc(0x451,0x10c,0x407,0x426)+_0x2af4bc(-0x2d4,-0x156,-0xd5,0x262)+_0x2af4bc(0x454,0x166,0x155,0x37a)+_0x2af4bc(0x2a1,0x220,0x41f,0x39b)+_0x26de55(0x251,0x481,0x1e4,0x409)+')\x20{\x0a\x20\x20\x20\x20\x20\x20'+'\x20\x20if\x20(inde'+'x\x20<\x20copied'+_0x2af4bc(0x185,0x251,0x319,0x342)+_0x26de55(0x2a5,0x1c8,0x207,0x2b4)+_0x26de55(0x41a,0x234,0x19c,0x53c)+_0x26de55(0x333,0x8a,0x48d,0x30c)+_0x26de55(0x86,0x39f,0x248,-0x62)+_0x26de55(-0x175,-0x1da,0x16,0x16c)+_0x26de55(-0xd2,0x141,-0x244,-0x25b)+_0x26de55(0x41a,0x72e,0x5bc,0x595)+_0x26de55(-0x127,-0x188,0xa4,-0x6b)+'l)\x20{\x0a\x20\x20\x20\x20\x20'+_0x26de55(-0xad,0x193,-0x2f9,-0x28e)+_0x2af4bc(0x2a2,-0xee,0x233,0xb6)+'rea\x20=\x20cell'+_0x2af4bc(-0x138,-0x1f3,0x63,-0x214)+_0x26de55(0x1ef,0x1cf,-0x105,0x4b5)+_0x2af4bc(0x3,-0x2d0,-0xd2,-0x7f)+_0x26de55(-0xad,-0x147,-0x86,-0x3b1)+_0x2af4bc(0x4c5,0x54c,0x3bb,0x6c5)+_0x2af4bc(0xfb,0x44c,0xe8,0xf5)+_0x2af4bc(0x39,0x1d3,0x24,-0x331)+_0x26de55(-0xad,-0x3ab,0x16e,-0x212)+'textarea.v'+_0x26de55(-0x108,-0xdb,-0x27f,-0x41a)+_0x2af4bc(0x32,0x1fb,0x140,0x226)+_0x2af4bc(0x5b3,0x235,0x4e4,0x81a)+_0x2af4bc(0x29a,0xac,0x1d4,0x50c)+_0x2af4bc(0x2c3,-0x1a0,0x24,-0x180)+_0x2af4bc(0x37c,0x50a,0x424,0x293)+_0x26de55(0x4a5,0x7fc,0x467,0x531)+_0x2af4bc(-0x34b,0x2e4,-0x27,0x39))+('row,\x20texta'+_0x2af4bc(-0xbf,0x2b7,0x297,0x5f2)+_0x26de55(0x1e6,0x493,0x2ba,-0x3d)+'iedColumnD'+_0x2af4bc(0x3fe,0x3a5,0x4e4,0x497)+');\x0a\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,0xac,0x11f,-0x1aa)+_0x2af4bc(0x602,0x82b,0x53a,0x23f)+_0x26de55(0x50a,0x6ac,0x62c,0x226)+_0x26de55(-0xad,0x110,0x3b,0x200)+_0x26de55(0x313,0x564,0x1fe,0x52f)+'\x20\x20\x20\x20\x20\x20\x20\x20}\x0a'+_0x2af4bc(-0x26d,0x245,-0x5,-0x177)+_0x26de55(0x51e,0x74f,0x816,0x7ac)+_0x26de55(-0x58,-0x235,-0x30a,0x2af)+_0x26de55(0x492,0x346,0x73d,0x2b8)+_0x2af4bc(0x63a,0x544,0x391,0x2bc)+_0x26de55(0x33d,0x59a,0x212,0x3dd)+'\x27\x20+\x20column'+_0x26de55(0x225,0x1cc,0x40d,0x279)+_0x26de55(-0x4e,0x136,-0x317,-0x276)+_0x2af4bc(-0xaa,0x157,0x1f3,0x52c)+_0x26de55(0xea,0x140,-0x25f,-0x120)+_0x2af4bc(0x2ff,0x54f,0x409,0x2c9)+_0x26de55(0x25a,0x20a,-0xeb,0x5b4)+_0x26de55(0xe8,0x20e,-0x197,0x3c8)+_0x2af4bc(0x8ce,0x73c,0x584,0x752)+'.add(colum'+_0x26de55(0xc3,0x361,0x2fb,0x181)+_0x2af4bc(0x7ff,0x3a0,0x58a,0x35f)+_0x26de55(0x497,0x351,0x707,0x5d9)+'document.g'+_0x26de55(0x506,0x85a,0x37a,0x83e)+_0x26de55(0x156,0x301,0x39c,0xa8)+_0x2af4bc(0x25d,0x1ad,-0xa5,-0x160)+_0x26de55(0x46a,0x6c0,0x4bb,0x36b)+_0x2af4bc(0x5cd,0x528,0x58d,0x5de)+_0x26de55(0x317,0x1bf,0x17f,0x341)+_0x26de55(0x2ee,-0x3f,0x27e,-0x5f)+_0x26de55(-0x22,-0x1d6,-0xb6,-0x37c)+_0x2af4bc(0x2f1,-0x106,0x81,0xfe)+_0x26de55(-0x16a,-0x41e,0x180,-0x21d)+_0x26de55(0xaf,0x1aa,0x74,-0xc)+_0x2af4bc(0x10,0x10e,0x33c,0x4c3)+_0x26de55(0x2c,-0xa3,0x2b8,0x38b)+_0x26de55(-0x6f,-0x254,-0x33,-0x160)+_0x2af4bc(0x159,0x1e5,-0xb9,-0x41c)+_0x2af4bc(0x2e5,0x161,0xf9,0xff)+'1;\x0a\x20\x20\x20\x20hea'+'ders.forEa'+_0x26de55(-0x141,-0x2ba,-0x479,-0x189)+'n(th,\x20inde'+_0x26de55(0x49a,0x607,0x29c,0x51c)+_0x26de55(-0x107,-0x179,0x68,-0x2f7)+'dataset.co'+_0x26de55(0x27d,0x45f,0x251,0x3f4)+_0x26de55(-0xb4,-0x3d8,-0x7e,-0x1ad)+_0x2af4bc(0x44a,0x4b1,0x22c,0x430)+_0x2af4bc(0x631,0x630,0x481,0x1e8)+_0x2af4bc(0x501,0x4a3,0x3fa,0x1f6)+_0x26de55(0x139,0x11b,0x474,-0x19c)+_0x26de55(-0xad,-0x21e,0x1a5,-0x22e)+_0x2af4bc(0x32,0x2a,0x324,0x54c)+_0x26de55(0x32b,0x37c,0x380,0x494)+'\x20\x27none\x27;\x0a\x20'+_0x2af4bc(0xf2,-0xda,0x202,0x3c9)+_0x26de55(0x169,0x25c,0x8b,-0x1b4)+_0x26de55(0xf9,0x137,0xec,0x2b0)+_0x2af4bc(0x311,0x25c,0x28f,0x2df)+_0x2af4bc(0x130,0x1bb,0x14,0x282)+_0x2af4bc(0x399,0x349,0x2d4,0x29a)+_0x2af4bc(0x6e9,0x7bb,0x527,0x5a9)+_0x2af4bc(-0x174,0x3f2,0x92,-0x217)+_0x2af4bc(0xde,0xd7,0x292,0x365)+'ll(\x27tr\x27);\x0a'+'\x20\x20\x20\x20\x20\x20\x20\x20ro'+_0x26de55(0x84,-0x243,0xce,0x47)+_0x26de55(0x34e,0x415,0x5c2,0x3d6)+'row)\x20{\x0a\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20v'+_0x2af4bc(0x2c5,0x590,0x41d,0x3a5)+'row.childr'+'en[columnI'+'ndex];\x0a\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20i'+_0x26de55(-0x1c,-0x1e9,-0xc2,0x14d)+_0x26de55(0x41a,0x33d,0x1e8,0x331)+_0x26de55(0x1fa,-0xcf,0x39,0x78)+_0x2af4bc(0x3ec,0x7c5,0x501,0x792)+_0x26de55(0x2d6,-0x26,0x396,0x104)+_0x2af4bc(0x2a5,0x583,0x476,0x1d7)+_0x26de55(-0xd6,0x242,-0x346,0x74)+_0x2af4bc(0x1c4,-0x198,0xb3,0x4)+_0x2af4bc(-0x18e,0x350,0x41,0x1b)+_0x2af4bc(-0x226,0x6d,0x79,-0x83)+_0x26de55(0x492,0x3dd,0x29a,0x5d4)+_0x2af4bc(0x79e,0x46f,0x4e3,0x5bf)+_0x26de55(0x2d0,0x3c4,0x1df,0x485)+'\x20columnNam'+_0x2af4bc(-0x1f0,0x32e,0x10d,0x18))+(_0x26de55(-0x94,-0x3c2,0xaa,0x28a)+'HiddenColu'+_0x26de55(-0x17a,0x1a,-0x1dc,0x13c)+_0x26de55(-0x7a,0x261,0x20e,-0x2a)+'enColumns.'+_0x26de55(0x4d8,0x50d,0x5d1,0x356)+')\x20{\x0a\x20\x20\x20\x20\x20\x20'+_0x26de55(0x316,0x557,0x4dc,0x52e)+'o\x20hidden\x20c'+'olumns\x27);\x0a'+_0x26de55(0xdd,0x29d,0x2df,0x168)+_0x2af4bc(0x3cf,0x37e,0x25b,0xcd)+'}\x0a\x20\x20\x20\x20\x0a\x20\x20\x20'+_0x2af4bc(-0x32,0x2c,-0xb9,-0xc9)+'nList\x20=\x20Ar'+_0x2af4bc(0x1ba,0x2f9,0x3a0,0x366)+_0x2af4bc(0x14e,0xe1,0x2a7,0x547)+_0x26de55(-0x16f,-0x1e0,0x19a,0xd3)+_0x2af4bc(0x18b,0x39c,0x1e2,0x44a)+_0x2af4bc(0x45d,0x407,0x4a5,0x432)+_0x26de55(0x4f1,0x2db,0x36d,0x774)+_0x26de55(0xb2,0x2a7,-0x29,0x88)+'\x20\x27\x20+\x20colum'+_0x2af4bc(0x516,0x38e,0x3f8,0x651)+_0x26de55(0x455,0x24d,0x2cc,0x3aa)+_0x26de55(-0x54,-0x38d,-0x321,0x1a4)+_0x2af4bc(0x33f,-0x2bd,0x2a,-0x50)+_0x2af4bc(0x5fb,0x354,0x447,0x77e)+_0x26de55(-0x12e,-0x34,-0x303,-0x35f)+');\x0a\x20\x20\x20\x20\x0a\x20\x20'+_0x2af4bc(0x19f,0x134,0x495,0x78d)+_0x2af4bc(0xb6,-0x191,-0xbb,-0x364)+_0x2af4bc(0x4e0,0x591,0x56f,0x603)+_0x2af4bc(0x474,0x3d4,0x286,0x1f9)+_0x26de55(0xd1,0x141,-0x17b,0xf3)+'}\x0a}\x0a\x0afunct'+_0x26de55(-0xf1,0x1e6,-0x1ab,-0x20e)+_0x26de55(-0x57,0x3f,0x176,0x1df)+_0x2af4bc(0x616,0x1c6,0x48e,0x1cc)+_0x26de55(0xba,-0x5b,0xee,-0x247)+_0x2af4bc(0x68b,0x42f,0x46c,0x640)+_0x26de55(0x2ad,0x157,0x317,-0x6a)+_0x2af4bc(0x39b,-0x230,0xd2,-0x16)+_0x2af4bc(0x348,0x94,0x265,0x51e)+_0x2af4bc(0x1b1,0x154,0xa2,0x3ad)+_0x26de55(0x4d0,0x535,0x209,0x4cb)+'cument.get'+_0x2af4bc(-0x1fa,0x2d4,0x27,0x10a)+'d(\x27tableBo'+_0x26de55(-0x1a8,-0x3a,-0x70,-0x153)+'\x0a\x20\x20\x20\x20var\x20h'+_0x2af4bc(0x7d2,0x303,0x475,0x5a0)+'eader.quer'+_0x26de55(0x1c1,0x4c4,0x12d,-0x6e)+'ll(\x27th\x27);\x0a'+'\x20\x20\x20\x20header'+_0x2af4bc(0x447,0x4d4,0x51b,0x881)+_0x26de55(0x30f,0x66f,0x1c3,0x137)+_0x26de55(-0x19a,0xaa,-0x230,0xc7)+_0x2af4bc(-0x38d,-0x2c7,-0x53,0x252)+_0x26de55(0x177,0x439,0x277,-0xbe)+'=\x20\x27\x27;\x0a\x20\x20\x20\x20'+_0x2af4bc(-0x5e,-0x13a,0x1d2,0x3bc)+_0x2af4bc(0x503,0x14b,0x1a4,0x3fc)+_0x26de55(0x24a,0x486,0x25,0x1f6)+_0x2af4bc(0x97,0x664,0x3e1,0x55b)+_0x26de55(-0x20,-0x2cb,0x2f2,0x63)+_0x2af4bc(0x236,0x50e,0x3d6,0x466)+_0x26de55(0x44a,0x20e,0x381,0x347)+_0x26de55(0x1c0,-0x121,0x230,0x445)+_0x26de55(0xd0,0x1d7,0xf0,0x27e)+_0x2af4bc(0x185,0x7f,0x2b0,0x484)+_0x2af4bc(-0x19b,0x19d,-0x96,0x178)+_0x2af4bc(0x467,0x29d,0x228,0x336)+_0x26de55(0x16f,0x121,0x4c7,-0x109)+_0x2af4bc(-0x261,0x159,-0x19,-0x7f)+_0x26de55(0x3b5,0xae,0x275,0x6c)+_0x26de55(0x2a3,0x5eb,0x1cc,0x6a)+_0x26de55(0x422,0x462,0x509,0x37f)+_0x26de55(0x108,0x470,-0x1c5,0x188)+_0x2af4bc(0x26a,0x374,0x5ab,0x3fe)+');\x0a\x20\x20\x20\x20});'+_0x26de55(0x2f9,0x42a,0x243,0xb6)+_0x2af4bc(0x267,0x4e4,0x195,0x443)+_0x26de55(-0x49,0x112,-0x3ad,-0x2f)+_0x2af4bc(0x698,0x446,0x520,0x414)+_0x26de55(0x492,0x30c,0x203,0x421)+'Message(\x27A'+_0x26de55(0x372,0x661,0x634,0x522)+_0x2af4bc(0x157,0x31b,0x1d5,-0x148)+_0x26de55(0x31,-0x236,0x326,-0x30d)+'\x20\x20\x20\x20\x20\x20\x20\x20va'+'r\x20currentD'+_0x2af4bc(0x2c3,-0x8c,0xe0,-0x179)+_0x2af4bc(0x5b4,0x507,0x363,0x4e1)+_0x26de55(0x383,0x504,0x2d6,0x1c0)+_0x2af4bc(-0x208,-0x21,0xd7,0x2fe)+_0x26de55(0x10a,-0x36,0x302,0x16e)+_0x26de55(0x267,0x469,0x439,0x37d)+_0x26de55(0x2b0,0x494,0x50e,0x351))+(_0x2af4bc(0x64d,0x35c,0x354,0x4d9)+_0x2af4bc(0xa7,0x465,0x1c1,-0x91)+'lRows\x20=\x200;'+_0x2af4bc(0x4a5,0x385,0x28d,0x4ab)+_0x2af4bc(-0xdc,0x244,0x1,0x352)+_0x26de55(0x449,0x19d,0x348,0x17a)+_0x26de55(0xae,-0xff,0x1f4,0x2d1)+'data\x20||\x20da'+'ta.length\x20'+_0x2af4bc(0x30d,0x49c,0x24e,0x27a)+_0x2af4bc(0x50c,-0x11d,0x1bf,0x119)+'\x20\x20\x20\x20curren'+_0x2af4bc(0x164,-0x2a8,-0xd,-0x260)+'ta;\x0a\x20\x20\x20\x20to'+'talRows\x20=\x20'+_0x26de55(0x1b7,0x462,0x426,0x16e)+_0x2af4bc(0x72f,0x405,0x3da,0x3cc)+'\x20\x0a\x20\x20\x20\x20var\x20'+_0x26de55(0x26d,0x59d,0x112,0x264)+'ocument.ge'+_0x26de55(-0xe0,-0x439,-0x3fd,0x15d)+_0x2af4bc(-0x15c,0x1fe,0xc9,0x189)+'eader\x27);\x0a\x20'+_0x2af4bc(0x5ef,0x4fa,0x570,0x8c2)+'y\x20=\x20docume'+'nt.getElem'+'entById(\x27t'+_0x2af4bc(0x2c5,-0x3d,0x314,0x514)+_0x26de55(0x35,-0x152,0x286,0x11e)+_0x26de55(0x1ab,0xb5,0x4d8,0x4ce)+'=\x20document'+_0x2af4bc(0x5b7,0x186,0x27e,0x50b)+_0x26de55(0x192,0x2a4,-0x16d,0x2f1)+'leContaine'+_0x2af4bc(0x238,0x2f,0x249,0x2eb)+_0x26de55(0x23f,0x591,0x19e,0x23f)+'.innerHTML'+'\x20=\x20\x27\x27;\x0a\x20\x20\x20'+'\x20body.inne'+'rHTML\x20=\x20\x27\x27'+';\x0a\x20\x20\x20\x20\x0a\x20\x20\x20'+_0x26de55(-0x18a,0x161,-0x238,0x159)+_0x2af4bc(0x4e1,0x350,0x280,0x562)+_0x2af4bc(-0x1eb,-0xc8,0x85,-0x1a4)+_0x26de55(0x3df,0x4a6,0x72d,0x2cc)+'d\x27];\x0a\x20\x20\x20\x20v'+_0x2af4bc(0x186,0x4b9,0x465,0x3b4)+'edColumns\x20'+'=\x20new\x20Set('+');\x0a\x20\x20\x20\x20var'+_0x2af4bc(-0x71,0x173,-0x64,-0x67)+_0x26de55(0x9b,-0xc4,0xf9,0x13c)+_0x26de55(0x2e1,-0x2e,0x336,-0x46)+_0x2af4bc(0xc1,0x9c,0x4b,-0x219)+_0x26de55(0x209,-0x10e,0xbb,0x130)+'ow\x20=\x20docum'+_0x26de55(0xac,-0x9a,-0xf2,0x2c6)+_0x26de55(0x1a3,0x39e,0x286,0x34f)+'r\x27);\x0a\x20\x20\x20\x20v'+_0x2af4bc(-0x222,-0x10d,0x129,0x51)+_0x26de55(0x231,0x2da,0x276,0x127)+_0x2af4bc(0x71b,0x28b,0x525,0x6de)+_0x2af4bc(0x220,0x377,0x260,0x226)+_0x26de55(0x450,0x3b8,0x5c9,0x555)+_0x26de55(-0x141,-0x31b,-0x24d,0x22)+_0x2af4bc(-0x1b3,-0xf7,-0x9,-0x16e)+'ex)\x20{\x0a\x20\x20\x20\x20'+_0x2af4bc(0x132,0x2f3,-0x65,-0x1ef)+_0x2af4bc(0x210,0x13b,-0x3b,-0x10d)+_0x2af4bc(0x557,0x530,0x5ee,0x59e)+_0x26de55(-0x15d,-0xfb,0x108,-0x22b)+_0x26de55(0xed,0x37b,0x36e,-0x234)+'(!columnsT'+'oHide.incl'+'udes(clean'+'Col)\x20&&\x20cl'+_0x26de55(0x14a,0x1e7,0x285,0x204)+_0x26de55(0x42d,0x365,0x489,0x1a1)+_0x26de55(0x260,0x1b8,0x35b,0x132)+_0x2af4bc(-0x36,0x1ea,0xc,0x24c)+'Indices.pu'+'sh(index);'+_0x26de55(0x41a,0x4be,0x6dd,0x59f)+_0x2af4bc(0x434,0x66f,0x3c5,0x537)+_0x26de55(0x28a,0x507,0x11a,0x42b)+_0x2af4bc(0x55e,0x51d,0x53d,0x492)+_0x2af4bc(0x1df,0x437,0x17d,0x4c5)+_0x26de55(0x1a3,0x44c,-0xc6,0x4f2)+_0x2af4bc(0x2a5,0x12b,0x3ba,0x1fb)+_0x26de55(0x244,0x36c,-0x97,0x191)+_0x26de55(0x1b2,-0x11,0x7d,0x7e)+_0x2af4bc(0x251,-0x2bf,-0xd0,0x250)+'\x20\x27\x27;\x0a\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20th.'+_0x2af4bc(0x381,0x21a,0x33b,0x1b2)+_0x26de55(0x34,-0x1ca,0x2fd,0x35e)+_0x26de55(0x35b,0x140,0x316,0x539)+_0x26de55(0x41a,0x3ad,0x1d5,0x196)+_0x26de55(0x43f,0xf6,0x62e,0x14c)+'set.column')+(_0x2af4bc(0x5c1,0x65b,0x53f,0x436)+_0x26de55(0x23a,0x4ae,-0xc7,0x50f)+_0x26de55(0x3a0,0x335,0x4c9,0x171)+_0x2af4bc(0x52b,0x17f,0x281,0x476)+_0x26de55(0x1f0,0x5c,-0x112,0x47a)+_0x2af4bc(0x225,0x596,0x239,-0x93)+_0x26de55(-0xad,0x276,0x22d,-0x1e2)+_0x26de55(0x250,0xb1,0x207,0x126)+_0x2af4bc(0x1ce,0x7a1,0x44b,0x61d)+_0x26de55(-0x2e,-0x2dc,-0x382,0x1d4)+'expand\x20col'+_0x2af4bc(0x4a8,0x5ea,0x290,0x20d)+'\x20\x20\x20\x20\x20\x20\x20\x20\x0a\x20'+_0x26de55(-0xad,-0x285,0x51,0x262)+_0x2af4bc(-0xf6,-0x1ab,-0x64,0x299)+'olumnWidth'+_0x26de55(0xb1,0x253,-0x139,0x399)+_0x2af4bc(-0xf3,-0x1c3,0x9a,-0x20e)+_0x2af4bc(0x243,-0xa0,0x105,0x2dc)+_0x26de55(-0x110,-0x2d4,-0x2ce,-0xe0)+_0x26de55(0x3be,0x6ea,0x181,0x68e)+'\x20\x20\x20\x20\x20\x0a\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20th'+_0x2af4bc(0x199,-0x119,0x199,-0xb9)+_0x26de55(0x15e,0x164,0x45a,0x209)+'e)\x20{\x0a\x20\x20\x20\x20\x20'+_0x2af4bc(-0x1ba,-0x217,0x24,-0x23b)+'\x20var\x20colNa'+_0x2af4bc(-0x348,-0x2ee,0x1f,-0x1da)+_0x26de55(0x4e9,0x55b,0x785,0x33a)+_0x2af4bc(0x17b,0x612,0x345,0x2a5)+_0x2af4bc(0xd8,0x2b7,0x24,-0x4c)+'\x20\x20\x20\x20\x20\x20if\x20('+_0x2af4bc(0x780,0x818,0x55a,0x8b3)+_0x2af4bc(0x4a2,0x2fd,0x589,0x4eb)+'(colName))'+'\x20{\x0a\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20collaps'+'edColumns.'+_0x26de55(0x4e3,0x3ae,0x2f4,0x6ce)+_0x26de55(-0x6,-0x17c,0x1fd,-0x347)+_0x2af4bc(0x189,-0x1b5,0x24,0xf)+'\x20\x20\x20\x20\x20\x20\x20thi'+_0x26de55(-0x23,0x22d,0x130,-0x24f)+_0x2af4bc(0x39b,0xb,0xbe,-0x8a)+_0x2af4bc(0x43c,0x6b9,0x5b3,0x488)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20}'+'\x20else\x20{\x0a\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x3b0,0x332,0x2be,0x44c)+'llapsedCol'+_0x2af4bc(0x38c,0x326,0x203,0x14f)+_0x26de55(0x4fb,0x638,0x5c7,0x7a5)+_0x26de55(-0xad,-0x378,0x12f,0x3e)+_0x2af4bc(0x71d,0x1a4,0x471,0x512)+_0x26de55(0x4f8,0x380,0x288,0x7ca)+_0x2af4bc(0x2bd,0x675,0x59f,0x450)+'\x270.5\x27;\x0a\x20\x20\x20'+_0x26de55(-0xad,-0x263,0x223,-0x31f)+_0x2af4bc(0x8b,0x30b,0x5a,0x2f2)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x237,0x506,0x1fa,0x1b8)+_0x26de55(0x39f,0xa0,0x3de,0x3cc)+_0x26de55(0x240,0x53e,-0x1f,0x39)+_0x26de55(0x3d8,0xf4,0x4a2,0x2e8)+_0x2af4bc(0x4b6,0x3a0,0x572,0x55c)+_0x26de55(0x0,0x2fa,0x2be,0x185)+'contextmen'+_0x26de55(-0x2d,-0xa9,0x32e,-0x78)+_0x26de55(0x439,0x5f4,0x562,0x4ce)+_0x26de55(-0xad,-0x10f,0x254,0x22f)+_0x2af4bc(0x4f2,0x6aa,0x547,0x282)+_0x2af4bc(0x254,0xaa,0x8e,0x8b)+_0x2af4bc(0xc3,0x554,0x35f,0x432)+'taset.colu'+_0x26de55(0x436,0x42d,0x32d,0x1d2)+_0x26de55(-0xad,-0x3d3,-0x3fc,0x111)+'\x20};\x0a\x20\x20\x20\x20\x20\x20'+_0x26de55(0x2b1,0x169,0x491,0x113)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20h'+_0x26de55(0x34d,0x34c,0x84,0x46d)+_0x26de55(0x28c,0x5a2,0x56,0x4fe)+_0x2af4bc(0x377,0x2e0,0xf4,0x2fa)+_0x2af4bc(0x3e2,0x5a8,0x3e4,0x1fc)+'});\x0a\x20\x20\x20\x20he'+_0x2af4bc(0x13a,0x3cb,0x1ef,-0x75)+_0x26de55(-0x3b,-0x315,0x8c,-0x1a)+_0x2af4bc(0x33b,0x1cd,0x12c,0x2d9)+'\x20\x20\x20\x0a\x20\x20\x20\x20fu'+'nction\x20upd'+_0x26de55(0x40b,0x55c,0x208,0x33b)+_0x26de55(-0x10d,0x111,0x23c,-0x46d)+_0x26de55(0x350,0x4ca,0x100,0x666)+'questAnima'+_0x2af4bc(0x56c,0x54d,0x538,0x784)+_0x26de55(-0x15e,-0x260,0x15c,-0x42b)+_0x26de55(0x3bf,0x29d,0x8c,0x2b2)+_0x2af4bc(0x880,0x8b5,0x565,0x616)+_0x2af4bc(0x302,0x2ec,0x1e1,-0x12c))+(_0x2af4bc(0x2e5,0x268,0x483,0x63d)+_0x26de55(0x1c9,0x40e,-0x19,0x464)+_0x2af4bc(0x31c,0x244,0x132,0x153)+_0x26de55(-0x69,-0x11,-0x3b,-0x18e)+'\x20headerRow'+_0x26de55(-0x6e,0x4f,0x2db,-0x1ce)+'ctorAll(\x27t'+'h\x27).forEac'+_0x26de55(0x29c,0x55e,0x399,0x2d1)+_0x2af4bc(0x5fa,0x339,0x29e,-0x9)+_0x26de55(0x1fe,0x102,0x3ea,0x34c)+_0x26de55(0x2e7,0x19c,0x492,0x5e3)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0x34,-0x182,-0x10e,0xb6)+_0x26de55(0x289,0xf6,0x278,0x4d4)+_0x26de55(0x278,0x5b3,0x28e,-0x90)+'et.columnN'+_0x26de55(-0xbb,-0x337,-0xf7,-0x248)+_0x26de55(0x75,-0xcf,-0x143,0x262)+_0x26de55(0x323,0x232,0x2e5,0x39d)+_0x2af4bc(-0x78,0x2cc,0x137,0x5c)+_0x26de55(0x4d1,0x34f,0x313,0x5a4)+_0x26de55(-0x14b,-0x3fb,-0x28f,-0x43c)+_0x2af4bc(-0x1da,0xd4,-0x5e,0x1c0)+'\x20\x20\x20\x20\x20\x20\x20\x20\x0a\x20'+_0x2af4bc(-0x1d,0x309,0x24,0x39)+'\x20//\x20Add\x20tr'+'ansition\x20i'+'f\x20not\x20alre'+_0x2af4bc(-0x99,-0x2cb,0x21,0x5)+_0x2af4bc(0x281,0x236,0x2c7,0xa5)+'\x20\x20\x20\x20if\x20(!h'+_0x2af4bc(0x527,0x1f0,0x3dd,0x487)+'style.tran'+_0x26de55(0xbf,0x7a,-0xe0,0x35d)+_0x2af4bc(0x302,0x344,0x24,0x17)+_0x2af4bc(0xec,-0x371,-0x24,-0x14e)+_0x26de55(0x514,0x544,0x34c,0x213)+_0x26de55(0x1b3,0x50,-0x17b,0x2bc)+_0x26de55(0x448,0x777,0x77e,0xe2)+_0x2af4bc(0x418,0x6b,0xb6,0xd3)+_0x2af4bc(0x6a9,0x722,0x5b3,0x278)+_0x2af4bc(0x231,0x32b,0x87,-0x3a)+_0x26de55(0x14c,-0x2,0x1df,0x188)+_0x26de55(-0xad,-0x4e,0x2c,-0x33a)+'\x20\x20if\x20(isCo'+'llapsed)\x20{'+'\x0a\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20hea'+'derCell.st'+_0x2af4bc(0x3d5,0x6fb,0x53e,0x48a)+_0x2af4bc(0x485,0x2c8,0x152,0x483)+_0x2af4bc(-0x2fe,0x159,0x24,0xd7)+_0x26de55(-0xf5,-0x3df,-0x3fe,0x46)+'erCell.sty'+'le.minWidt'+'h\x20=\x20\x2730px\x27'+_0x2af4bc(0x5d,0x273,0x1d4,0xb1)+_0x26de55(0x4bb,0x297,0x55e,0x67b)+_0x2af4bc(0x318,0x5e0,0x4d1,0x38b)+_0x2af4bc(0x3bc,-0x24d,0xf2,-0x172)+_0x2af4bc(0x353,0x25e,0x178,0x22a)+_0x26de55(0x1bb,0x51,-0x140,-0xfa)+_0x2af4bc(-0x2b7,-0x30c,0x24,0x273)+_0x2af4bc(0x709,0x5b5,0x545,0x452)+_0x26de55(0x42b,0x2fb,0x3c8,0x518)+'ding\x20=\x20\x272p'+_0x2af4bc(0x210,0x516,0x28c,-0xb2)+_0x2af4bc(-0x40,0xbd,0x24c,0x20f)+_0x2af4bc(0x839,0x87a,0x55c,0x624)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x2c2,0x288,0x401,0x222)+'l.style.wi'+_0x2af4bc(0x368,0x40d,0x161,0x458)+_0x2af4bc(-0x1a8,-0x1,0x24,0x282)+_0x2af4bc(-0x369,0x1d3,-0x24,-0x223)+_0x26de55(0x514,0x252,0x6c5,0x773)+'le.minWidt'+_0x26de55(-0x119,-0x8d,-0x26f,-0xc5)+_0x2af4bc(-0x113,0x26b,0x24,0x1f3)+_0x2af4bc(0x595,0x4f7,0x310,0x398)+_0x26de55(-0x130,0x16,0x223,0x12f)+_0x2af4bc(0x318,-0x12a,-0x26,0x274)+_0x26de55(0x484,0x51b,0x2dc,0x460)+_0x2af4bc(-0xde,-0x97,0x24,-0x143)+_0x26de55(0x370,0x5e,0x3fc,0xa3)+_0x2af4bc(0x46a,0x4d9,0x296,0x32b)+_0x26de55(0x478,0x255,0x318,0x674)+_0x2af4bc(-0x14e,0x31c,0x29,-0x120)+_0x26de55(0x3d8,0x274,0x223,0x644)+'\x0a\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x0a\x20\x20'+_0x26de55(-0xad,-0x367,-0x2af,-0x61)+'\x20\x20\x20\x20allRow'+_0x2af4bc(0x2ab,0x27a,0x51b,0x3cf)+'function(r'+'ow)\x20{\x0a\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20var\x20'+_0x2af4bc(0x161,0x1be,-0xac,-0x265))+('.children['+_0x26de55(-0x6d,0x1c4,0x8e,-0xc1)+_0x26de55(-0x13b,-0x224,-0xe2,-0x181)+_0x26de55(-0xad,0x204,0x13b,0xf5)+_0x2af4bc(0x3e9,0x1f3,0x505,0x508)+_0x26de55(-0xd1,0xf9,-0x172,-0x337)+_0x2af4bc(-0x2da,-0x1fc,0x24,0x2b7)+_0x2af4bc(-0x1db,0x17a,0x24,-0x273)+_0x26de55(0x2ba,0x39f,0x209,0x57)+'ea\x20=\x20cell.'+_0x2af4bc(0x5d5,0x63a,0x483,0x194)+_0x2af4bc(0x36a,0x857,0x577,0x2af)+_0x26de55(0x19c,0x2e4,0x1bc,0x3bb)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x1ca,-0x1c2,0x24,0x2e4)+_0x26de55(0x23c,0x126,0x5e,0x3ee)+_0x2af4bc(-0x281,-0x3b,0xd,0x2e1)+_0x2af4bc(-0x21d,0x8b,0x24,-0x2d9)+_0x2af4bc(0x1e,0x7a,0x24,-0x33)+_0x26de55(0x32f,0xa4,0x485,0x257)+_0x2af4bc(0x5c9,0x287,0x5ea,0x5df)+_0x2af4bc(0x1ce,-0x2e0,0x6b,0x281)+_0x26de55(0x349,0x515,0xc7,0x295)+_0x2af4bc(-0x27,0x1e1,0x24,0x307)+_0x2af4bc(0x2f4,-0x28,0x24,-0x214)+'\x20\x20\x20cell.st'+_0x26de55(0x21e,0x158,0x3a7,0x361)+'th\x20=\x20\x2730px'+'\x27;\x0a\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x385,-0x2ad,0x24,0x285)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20cell.styl'+_0x2af4bc(0x504,0x358,0x222,0x43d)+_0x26de55(-0x146,0x162,0x56,-0x2c9)+_0x2af4bc(0x73b,0x27e,0x4eb,0x2c4)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x1b4,0x6c1,0x481,0x29e)+_0x26de55(0x49,-0x30c,0x20a,-0x192)+_0x2af4bc(0x56b,0x61a,0x4e0,0x6a4)+_0x26de55(-0xfd,-0x3ad,0xba,0x1b8)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,-0x189,0x20a,0x266)+_0x26de55(0x3f6,0x423,0x3e6,0x177)+_0x26de55(0x1fd,0x29d,0x542,0x346)+_0x2af4bc(0x68d,0x693,0x43a,0x4c4)+_0x26de55(0xca,0x361,0x33,-0x1e0)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x797,0x64d,0x480,0x64d)+_0x2af4bc(0x60c,0x508,0x47d,0x452)+_0x2af4bc(0x43a,0x2d1,0x4eb,0x534)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20textare'+_0x2af4bc(0xfb,0x574,0x267,0x165)+_0x26de55(0x2d6,0x4d2,0x4b9,0x13c)+_0x2af4bc(0x727,0x44a,0x476,0x2a0)+_0x2af4bc(0x136,0x244,0x24,0x168)+_0x26de55(-0xad,0x98,-0x1fd,-0x3f)+_0x26de55(0x313,0x3df,0x54f,0x1c1)+_0x2af4bc(0x1de,-0xfd,0x24,0x9f)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x3dc,0x238,0xc1,0xf1)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,-0x3c2,0x144,-0x4)+_0x2af4bc(0x454,0x24e,0x2cb,0x1b8)+_0x26de55(0x11c,0xee,-0xbc,-0x74)+'dth\x20=\x20\x27\x27;\x0a'+_0x2af4bc(0x176,-0x1a1,0x24,-0x237)+_0x2af4bc(-0x215,0x12f,0x24,0x1ee)+_0x26de55(0x32f,0x3ff,0x2b1,0x549)+'ll.style.m'+_0x2af4bc(-0x343,0x27c,-0x7b,0xca)+'\x27\x27;\x0a\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,-0x32c,0x13b,0x8b)+_0x26de55(0x182,0x2a7,0x4e1,0x129)+_0x26de55(-0x19b,-0x303,0x1b8,-0x2df)+_0x2af4bc(0xe4,0x2d8,-0x48,0x1c)+_0x26de55(-0xad,0x1eb,-0x18f,0x46)+_0x2af4bc(-0x18b,-0xb8,0x24,0x1d8)+'\x20\x20\x20\x20\x20\x20cell'+_0x2af4bc(0x454,0x80a,0x4fc,0x83a)+_0x2af4bc(-0xc3,0x76,0x97,0x1f5)+'x\x27;\x0a\x20\x20\x20\x20\x20\x20'+_0x2af4bc(-0x243,-0x149,0x24,0x32f)+_0x2af4bc(0x9d,-0x9b,0x24,-0x171)+_0x26de55(0x182,0xc3,0x325,-0x177)+_0x26de55(-0x99,-0x2e5,0x2c6,0x1d0)+_0x26de55(-0x4b,-0x189,0x1f,0x7f)+_0x2af4bc(-0x335,-0xc,0x24,-0x2af)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x208,-0x12b,-0x50,0xcf)+_0x2af4bc(0x704,0x8c0,0x564,0x475)+_0x2af4bc(0x39d,0x69,0x2d4,0x2a)+_0x2af4bc(-0x277,-0x28b,0x24,0x2bd)+_0x26de55(-0xad,0x8,-0x138,0x197)+_0x26de55(0x513,0x338,0x7bc,0x371)+_0x2af4bc(-0x3a7,0xcb,-0x4f,-0x54)+'isplay\x20=\x20\x27')+('\x27;\x0a\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(-0x21b,-0x288,0x24,-0x22b)+_0x2af4bc(-0x3a,-0xb5,0x24,-0x188)+_0x2af4bc(0xd7,0x197,0xa6,-0x141)+_0x2af4bc(-0xeb,0x20b,0x24,-0x29c)+'\x20\x20\x20\x20\x20\x20\x20}\x0a\x20'+_0x2af4bc(0x343,0x2b3,0x24,-0x18f)+_0x26de55(0x3d8,0xc6,0x50a,0x60a)+_0x2af4bc(0x385,0x34a,0x4eb,0x1eb)+_0x26de55(0x2f,-0x332,-0x308,-0x8b)+_0x26de55(0x41a,0x76c,0x4b7,0x23a)+_0x2af4bc(0x6b,0x155,0x23a,0x3b0)+'\x20\x20\x20\x20\x20});\x0a\x20'+_0x2af4bc(0x3a6,0x34c,0x348,0x7e)+_0x2af4bc(0x24b,0x354,0x143,-0x7b)+_0x2af4bc(0x598,0x804,0x561,0x45d)+_0x26de55(-0xb9,0x1ce,-0x3ba,0x1fb)+_0x26de55(0x4db,0x824,0x196,0x4e6)+_0x26de55(-0x50,-0x168,0xd5,0x1bc)+_0x2af4bc(-0x1da,-0x199,-0x84,0x1ef)+'r.onscroll'+_0x26de55(0x3bb,0x323,0x82,0x29a)+_0x2af4bc(0x92,0x400,0x2f5,0x255)+_0x2af4bc(0x27b,0x4c0,0x597,0x383)+'rollTop\x20=\x20'+_0x2af4bc(0x436,0x6f,0x1ba,0x2ea)+_0x26de55(0x3fb,0x31b,0x643,0x1e2)+'\x0a\x20\x20\x20\x20\x20\x20\x20\x20v'+_0x26de55(0x2aa,0x4c8,0x133,0x458)+_0x26de55(0x404,0x315,0x63e,0x51c)+_0x26de55(-0x79,-0x75,-0x112,0x25)+_0x2af4bc(0x40b,0x18b,0xe6,0x281)+_0x26de55(0x155,0x39,0x34f,-0x101)+_0x2af4bc(0x9c,0x276,0x146,-0x1c2)+_0x26de55(-0x96,0xa6,0x295,-0x3d6)+_0x26de55(0x9f,0x1d2,0x21b,-0x7b)+_0x2af4bc(0x6a0,0x4e5,0x554,0x38f)+_0x26de55(0x2f4,0x3b6,0x466,0x49a)+_0x26de55(0x20d,-0x99,-0xac,0x29d)+_0x26de55(0x237,0x56f,0x4b1,-0x68)+_0x26de55(-0xfc,-0x59,-0x11d,0x5e)+_0x26de55(-0x1b1,0xf2,0x145,-0x221)+_0x2af4bc(-0x1f5,-0x311,0x24,-0x2a1)+_0x26de55(0x4c0,0x68b,0x478,0x37a)+_0x26de55(0x259,0xc1,0x440,0x36d)+_0x2af4bc(0xa6,0x591,0x23e,0x2dc)+_0x26de55(-0xad,-0x1d,0x11b,-0x3a5)+'visibleRow'+_0x26de55(-0x4d,0x227,-0x358,-0x9a)+'nd;\x0a\x20\x20\x20\x20\x20\x20'+_0x26de55(0x8a,0x205,0x2b9,0xc)+'erVisibleR'+_0x2af4bc(0x345,0x72b,0x5a4,0x61c)+_0x26de55(0x36b,0x207,0xb3,0x564)+'es);\x0a\x20\x20\x20\x20\x20'+_0x26de55(0x2e5,0x54b,0x150,0x5bc)+_0x2af4bc(0x4a3,0x582,0x2e9,0x343)+_0x26de55(0x241,0xc6,0x256,0x26a)+_0x2af4bc(0x370,0x681,0x33a,0xdd)+'eRows(vali'+_0x26de55(-0x111,-0x12d,-0x45a,-0x2bd)+_0x26de55(0x4f6,0x3bd,0x1a5,0x733)+_0x2af4bc(0x48,0x583,0x35b,0x85)+_0x2af4bc(0x7e,0xd3,-0x8e,0x9d)+_0x26de55(0x20c,0x1,0x3e7,0x109)+'lementById'+_0x2af4bc(0x232,0x4f3,0x235,0x397)+'y\x27);\x0a\x20\x20\x20\x20\x20'+'\x20\x20\x20var\x20hig'+_0x26de55(0x3b8,0x543,0x89,0xfa)+'wIndex\x20=\x20n'+'ull;\x0a\x20\x20\x20\x20i'+_0x26de55(0x4a3,0x5f3,0x75b,0x53c)+_0x26de55(0x3b8,0x2c1,0x10d,0x3ce)+_0x26de55(-0xa9,-0xb6,-0x39,0xd0)+_0x26de55(-0x11e,-0x32a,-0xeb,0x22d)+_0x26de55(0xb4,0x204,0xca,0x159)+_0x2af4bc(0x316,0x474,0x23c,-0x114)+_0x26de55(-0x6a,0x1f6,0x2d8,-0x1e7)+_0x2af4bc(0xca,0x2d6,0x3a2,0xba)+_0x26de55(0x150,0x41e,-0x30,0x1fa)+_0x26de55(-0xfe,0x16b,0x206,-0x382)+_0x2af4bc(0xdf,0x1cf,0x41,-0xc1)+'\x20\x20\x20\x20\x20\x20body'+_0x26de55(-0x76,-0x3ca,0x1fa,-0x1aa)+_0x26de55(0x37f,0x3ec,0xa9,0x425)+_0x2af4bc(0x18d,0x6ea,0x438,0x1fe)+_0x2af4bc(0x296,0x490,0x2f2,0x423)+'agment\x20=\x20d'+_0x2af4bc(0xce,0x4ec,0x28b,0x388)+_0x26de55(0x94,-0x2b9,-0x217,-0xe0)+'ntFragment'+_0x2af4bc(0x86,0x22f,0x3af,0x3f0)+'\x20\x20var\x20data'+'\x20=\x20current'+_0x26de55(0x407,0x38e,0x514,0x60a)+_0x26de55(-0x65,-0x1c9,0x48,0x146)+'\x20\x20\x20var\x20spa'+_0x2af4bc(0x655,0x46,0x39e,0xae)+_0x26de55(0x1ba,0x410,-0x135,0x348))+(_0x26de55(-0x152,-0xa5,-0x3d6,-0x4b9)+'t(\x27tr\x27);\x0a\x20'+'\x20\x20\x20\x20\x20\x20\x20spa'+_0x2af4bc(0x295,0x104,0xe9,0x2b9)+'le.height\x20'+'=\x20(visible'+'RowStart\x20*'+_0x2af4bc(0x307,0x378,0xf,-0x2dd)+_0x26de55(0x44d,0x2c0,0x42c,0x6e8)+_0x26de55(0x409,0x567,0x6b0,0x708)+_0x26de55(0xb5,-0x42,-0x31,0x399)+_0x26de55(0x332,0x3a1,0x323,0x46a)+_0x26de55(0x19,0xb9,0x12c,0x229)+_0x26de55(0x364,0x98,0x7c,0x56d)+_0x26de55(0x136,0x97,0x1a9,-0x16b)+_0x26de55(0x398,0x5ce,0xf5,0x4b6)+_0x2af4bc(0x2e0,0x389,0x25e,0x94)+_0x26de55(0x64,-0xa1,-0x2f8,-0x163)+_0x26de55(0x4a2,0x34a,0x554,0x2c8)+_0x26de55(0x331,0x494,0x4b6,0x72)+_0x2af4bc(0x39b,0x4e4,0x5b9,0x852)+_0x2af4bc(0x53,0x4cc,0x270,0x287)+_0x2af4bc(0x9,-0x326,-0x9c,0x11e)+_0x2af4bc(0x396,0x41,0x46,0x247)+_0x26de55(0x2e8,0x293,-0x3c,0x419)+_0x2af4bc(0x77,-0xb1,0x16a,0x4c8)+'\x20\x20\x20\x20\x20\x20\x0a\x20\x20\x20'+_0x2af4bc(0x355,0x799,0x478,0x1ff)+'var\x20i\x20=\x20vi'+'sibleRowSt'+'art\x20+\x201;\x20i'+'\x20<=\x20Math.m'+'in(visible'+'RowEnd,\x20to'+_0x2af4bc(-0x2da,0x65,-0xbe,0x163)+_0x26de55(-0x18d,0x89,0x199,-0x454)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20v'+'ar\x20row\x20=\x20d'+_0x26de55(0x1ba,0x2af,0x11f,0x385)+_0x26de55(-0x152,0xf7,0x18c,-0xd3)+_0x2af4bc(0x539,0x1ea,0x21f,0x56b)+_0x2af4bc(-0x1a6,0x222,0x24,0x2e1)+_0x2af4bc(0x317,0x3d2,0x359,0xbe)+_0x26de55(-0x1af,-0x157,0x175,-0x2cf)+_0x26de55(-0x169,-0x187,-0x1be,0xc4)+_0x2af4bc(0x2a4,-0xaf,0x24,0x2ac)+_0x2af4bc(0xd7,-0x2e7,-0x86,0x100)+_0x26de55(-0x132,0xd4,0x1e8,-0x322)+'dex\x20!==\x20nu'+_0x2af4bc(0x6b1,0x26e,0x4dd,0x3a8)+_0x2af4bc(-0x15a,0x24c,-0xc3,-0x364)+'tedRowInde'+'x)\x20{\x0a\x20\x20\x20\x20\x20'+_0x2af4bc(-0x325,0x19a,0x24,-0x290)+_0x26de55(0x35a,0x47c,0x365,0x29e)+_0x26de55(-0xdd,-0x32d,-0x2b4,-0x1e8)+'highlighte'+_0x2af4bc(0x1cc,0x4f6,0x337,0x1f4)+_0x2af4bc(0x130,0xae,0x24,0x1b4)+_0x2af4bc(0x5fa,0x59b,0x2cd,0x3ab)+'ightedRow\x20'+'=\x20row;\x0a\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20}'+_0x26de55(0x41a,0x1dd,0xe9,0x39d)+_0x2af4bc(0x1c9,0x4ee,0x3c5,0x1a4)+_0x2af4bc(0x2ae,0x555,0x2d9,0x123)+_0x26de55(0x4af,0x385,0x615,0x1d8)+'continue;\x0a'+_0x26de55(-0xad,-0x35b,-0x176,0x131)+_0x26de55(-0x69,-0x7f,-0x2cb,-0xf3)+_0x26de55(0x204,0x440,-0x50,0x239)+'ColumnIndi'+_0x2af4bc(0x29,0x2da,0x17c,0x4c1)+_0x26de55(0x29c,0x564,-0x1a,0x3e3)+_0x26de55(0x1d3,-0x157,0x270,0x1f8)+_0x2af4bc(0x440,0x2a6,0x32f,0x1ab)+_0x2af4bc(0x12a,0x6d9,0x486,0x30f)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'var\x20cell\x20='+_0x2af4bc(-0x35,0x159,0x200,0x418)+_0x26de55(-0x9f,0xeb,-0xe4,0x25d)+_0x2af4bc(-0x242,-0xaf,0x24,0x26e)+_0x2af4bc(0x68a,0x445,0x35b,0x169)+_0x2af4bc(-0x200,-0x2f2,-0x35,0x1c8)+_0x26de55(-0xec,-0x1a4,0x1d2,0x161)+_0x26de55(-0x81,0x167,-0x296,-0x319)+_0x2af4bc(0x633,0x539,0x372,0x4d9)+_0x2af4bc(0x128,0x124,-0x8c,-0x39e)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x367,0x626,0x3b,0x5e0)+_0x26de55(-0xad,0x12f,-0x78,-0x3a5)+'\x20\x20var\x20td\x20='+_0x26de55(0x18d,0x155,0x47c,0x47b)+_0x2af4bc(0xe7,0x51d,0x32c,0xc9)+_0x26de55(0x107,0x1c0,0x22a,0x3b0)+_0x2af4bc(0x4ff,0x5b0,0x4eb,0x476)+_0x26de55(0x75,-0x1f9,-0x2a9,0x167)+_0x26de55(0x363,0x241,0x2f0,0x324)+_0x26de55(0x3f3,0x216,0x59c,0x5ce)+_0x2af4bc(0x1f0,0x6c,0x369,0x360))+(_0x26de55(-0x98,-0xf2,-0x200,-0x198)+_0x2af4bc(-0x29f,0x243,-0xd2,-0x22e)+_0x26de55(-0xad,-0x1f8,-0x178,-0x100)+_0x2af4bc(0x841,0x6ad,0x5e4,0x933)+_0x26de55(-0xd3,-0x3b3,0x102,-0x77)+_0x26de55(-0x10e,0x9b,0x1bd,-0x5)+_0x26de55(0x4e2,0x6ab,0x5bf,0x3ac)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20t'+_0x26de55(0x187,0x2a7,0x1fc,-0x17e)+_0x2af4bc(0x518,0x257,0x27f,0x3b0)+'=\x20i;\x0a\x20\x20\x20\x20\x20'+_0x26de55(-0xad,-0x1de,0x288,0x11)+_0x2af4bc(0x10c,-0x178,0x173,-0x8c)+_0x2af4bc(0x593,0x4f6,0x5ba,0x83b)+'l\x20=\x20colInd'+'ex;\x0a\x20\x20\x20\x20\x20\x20'+_0x2af4bc(-0x186,-0x18a,0x24,0x297)+_0x26de55(0x464,0x1da,0x392,0x565)+_0x2af4bc(-0x20,0xf,0x177,0x28f)+_0x26de55(0x44c,0x424,0x5a3,0x2c9)+_0x2af4bc(0x1e5,0x3c2,0x1d4,0xd)+_0x26de55(-0x30,0x34,0x1b,-0x180)+'xtarea.sty'+_0x2af4bc(0x1f3,0x577,0x212,-0xa4)+_0x26de55(0x201,0x2b,-0x6f,0x361)+_0x26de55(-0xad,0x20d,-0x3e,-0x2b7)+'\x20\x20\x20\x20\x20\x20text'+_0x26de55(0x3e1,0x41e,0x11c,0x4a8)+_0x26de55(-0x9b,-0x11a,0x160,0x84)+_0x26de55(0x2c8,0x7a,0x3f3,0x2ec)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x1e5,0x529,0x31e,0x451)+_0x26de55(0x3e1,0x1de,0x129,0x55a)+_0x26de55(0x9e,0x2fc,0x3ec,0x3d5)+_0x26de55(-0xba,0x24b,-0x1f6,-0x1fd)+_0x26de55(0x2db,0x5ad,0x470,0x1b4)+_0x26de55(-0xad,-0x30a,-0xbe,-0x1fd)+_0x2af4bc(0x3bf,0x83,0x1ab,0x3c5)+_0x26de55(0x26e,0x1c6,-0x37,0x2a6)+_0x26de55(0x1e0,-0x15,-0x12e,-0x78)+_0x26de55(0x1c,0x303,-0x2cb,-0x306)+_0x2af4bc(0x2a9,0x1fe,0x1d7,0x9c)+_0x26de55(0x4aa,0x7fa,0x2dd,0x41f)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x464,0x48b,0x169,0x2ce)+_0x2af4bc(0x175,0x8f,0x174,0x1a6)+_0x2af4bc(0x62f,0x1bb,0x453,0x349)+_0x2af4bc(0x483,0x53e,0x5b3,0x7e9)+_0x26de55(0x3a0,0x434,0x51d,0x62b)+_0x26de55(-0x70,0x2f9,0x126,-0x71)+_0x26de55(-0x5a,-0x393,-0x10a,-0x14)+_0x26de55(0x343,0x365,0x1ad,0x2b6)+'4\x27;\x0a\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,-0x42,0x85,0x14)+'textarea.s'+_0x2af4bc(0x12a,0x191,0x299,0x132)+_0x26de55(-0x97,0x31,0x222,-0x2bc)+'o\x27;\x0a\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x464,0x71c,0x2e9,0x5af)+_0x26de55(-0x14a,-0x21d,-0x251,-0x250)+_0x2af4bc(-0x8d,-0x185,0x188,0x1c0)+'rder-box\x27;'+_0x26de55(0x41a,0x169,0x121,0x3e9)+'\x20\x20\x20\x20\x20\x20\x20tex'+_0x2af4bc(0x813,0x724,0x4b3,0x45f)+_0x26de55(0x4cd,0x3ee,0x70a,0x5bf)+'\x27100%\x27;\x0a\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x21f,-0x225,0x6c,0x267)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x19d,0xcb,-0x156,-0x1b)+_0x2af4bc(0x1b8,0x5df,0x2d1,0x62f)+_0x26de55(0x50b,0x560,0x2f8,0x81f)+_0x26de55(0x3c7,0x58c,0x1c7,0x594)+_0x2af4bc(-0x211,0xe,0x24,0x297)+_0x2af4bc(-0x49,0x35b,0x24,0x259)+_0x2af4bc(-0x6d,-0x298,-0x2e,0x25c)+_0x2af4bc(0x33d,0x3ed,0x1b4,0x3b)+_0x2af4bc(0x694,0x3a4,0x4c9,0x5c8)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0xe0,0x238,0x5a,-0x28e)+_0x26de55(-0xad,-0x363,-0x104,-0x3be)+_0x26de55(0x516,0x73e,0x476,0x205)+_0x2af4bc(-0xde,0x1be,-0x39,-0x2c9)+'\x20(cleanCol'+_0x2af4bc(0x338,0x90a,0x5af,0x73c)+_0x2af4bc(0x68d,0x60a,0x422,0x55e)+_0x2af4bc(0x21e,0x16b,0x24,0x18d)+_0x2af4bc(0x5c5,0x541,0x471,0x571)+_0x2af4bc(0x301,0x28c,0x61,-0xa2)+_0x26de55(0xbb,-0x155,0x1bf,-0x256)+_0x26de55(0x71,0x24,0xb1,0xed)+_0x2af4bc(0x1c9,0x346,0x4eb,0x2d6)+_0x2af4bc(-0x18a,-0x2d,-0x8a,0x51)+_0x2af4bc(0xf0,0x2eb,0x2,0x201)+_0x26de55(-0xad,-0x2a4,-0x340,0x129)+_0x26de55(0x24d,0xe,0x4dd,-0x4e)+_0x2af4bc(0x3a3,0x3ca,0x4b2,0x49f)+_0x26de55(0x1ca,0x3cb,0x6c,0x5f))+('\x27vertical\x27'+_0x2af4bc(0x7f,-0xd7,0x1d4,0xf5)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0x69,-0xd,0x2c5,-0x328)+_0x26de55(-0xad,-0x159,-0xfd,0x186)+'\x20\x20\x20var\x20las'+'tHeight\x20=\x20'+_0x2af4bc(0x332,0x3ea,0x26c,0x1c)+_0x26de55(-0xad,0x12e,0x72,0x1df)+_0x26de55(0x367,0x41e,0x5d8,0x2a)+_0x26de55(-0xad,-0x363,-0x281,-0x227)+_0x2af4bc(0x2e4,0x4a6,0x31e,-0x35)+_0x26de55(0x452,0x2ff,0x3b6,0x3ff)+_0x26de55(0x4f0,0x295,0x772,0x3e4)+_0x2af4bc(0x173,0x31a,0x164,0x3a3)+'wn\x27,\x20funct'+_0x2af4bc(0x2ec,0xc9,0xbb,0x9a)+_0x2af4bc(-0xde,-0x1a4,0x24,0x1a1)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20var\x20rec'+_0x26de55(0x3fc,0x743,0x1ae,0x610)+'etBounding'+_0x2af4bc(0x36e,0x73c,0x5d2,0x3b7)+_0x2af4bc(0x250,0x1c2,0x3af,0x6c9)+_0x26de55(-0xad,0x7c,-0x297,0x110)+_0x2af4bc(0x2ce,0x1fc,0x207,0x271)+'r\x20isNearBo'+'ttomRight\x20'+_0x2af4bc(0xac,0x44d,0x30c,0x45e)+_0x26de55(-0xad,-0x42,-0x259,-0x261)+_0x26de55(-0xad,0x139,0x9d,0x290)+'\x20\x20e.client'+_0x2af4bc(0x2a7,0x3b2,0x5cb,0x297)+'ight\x20-\x2020\x20'+'&&\x0a\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,-0xb4,0x2b5,-0x3ef)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x47,0x235,0xc3,0xf7)+_0x2af4bc(0x348,0x3e2,0x54c,0x81b)+_0x2af4bc(0x422,0x79,0x3bd,0x521)+_0x26de55(-0xad,-0x40f,-0x194,-0x246)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x71b,0x6ed,0x5c3,0x7bb)+_0x26de55(-0xad,0x15b,-0x3b8,0x58)+_0x26de55(-0xad,-0x2fb,-0x11,-0x1ef)+'\x20\x0a\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20if\x20('+_0x26de55(0x42,0x2c8,-0x12a,0x4b)+_0x2af4bc(0x131,-0x5e,-0x85,0x1df)+_0x2af4bc(0x807,0x3bf,0x4eb,0x3a9)+_0x2af4bc(0x23b,-0x31d,0x24,0x17a)+_0x26de55(0x260,0x171,0x55b,0x215)+_0x26de55(0x509,0x68e,0x28f,0x693)+_0x2af4bc(0x405,0x64b,0x394,0x6a1)+_0x2af4bc(0x2ce,0x1f6,0x21f,0x157)+_0x26de55(-0xad,-0x2fa,-0x189,0x28e)+_0x2af4bc(0x1e7,-0x2fb,0x24,-0x244)+_0x26de55(0x75,0x151,0xd2,-0x274)+_0x26de55(0x2fb,0x3a1,0x47,0xd8)+_0x26de55(0x365,0x5ec,0x63b,0x3c7)+_0x26de55(-0xad,0xc7,-0x1a8,-0x10e)+_0x26de55(-0xad,0xaa,0x89,-0x84)+_0x2af4bc(-0x17e,-0x294,0x68,-0x186)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x2a4,0x240,0x24,0x2ab)+_0x26de55(0x485,0x48d,0x15d,0x540)+_0x2af4bc(0x346,-0x25d,0x13,0x282)+_0x26de55(-0x17b,-0x257,-0x349,0x17e)+_0x26de55(0x1f9,0x16b,0x24a,0xcc)+_0x26de55(-0x15c,-0x38,0x25,-0x9a)+_0x2af4bc(-0xf2,0xb3,0x24,0x315)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(-0x1bd,-0x27d,0x24,-0x25d)+_0x2af4bc(0x27a,0x4be,0x36e,0x2d2)+'tHeight\x20=\x20'+_0x2af4bc(0x456,0x32c,0x166,-0xa6)+_0x2af4bc(0x342,-0x38,0x65,0x386)+_0x2af4bc(-0x2ca,-0x15b,0x24,0x2eb)+_0x26de55(-0xad,-0x150,0x9b,-0x17b)+_0x2af4bc(-0x160,0x19,0x24,-0x333)+_0x2af4bc(-0x247,-0x4b,0xc8,-0x1ee)+_0x2af4bc(-0xb7,0x43d,0x108,0x34c)+'==\x20lastHei'+_0x26de55(-0x192,-0x320,-0x31e,-0x4cb)+_0x2af4bc(0x1d7,0x30,0x24,-0x4e)+_0x2af4bc(0x26e,-0x2a6,0x24,-0x2ae)+_0x2af4bc(0x100,0x75,0x24,-0x55)+_0x2af4bc(0x49d,-0x7a,0x14d,0x3b9)+'ght\x20=\x20curr'+_0x2af4bc(0x546,0x593,0x4de,0x49a)+_0x2af4bc(0x53e,0x29c,0x4eb,0x5bf)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,0x233,-0xe5,0xe4)+_0x2af4bc(0x405,-0x160,0x1be,0x463)+_0x2af4bc(-0x94,0x152,-0xaf,-0x170)+_0x26de55(-0xad,-0x1dc,-0x354,0x175)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x634,0x5f8,0x4d7,0x70c))+('w.querySel'+_0x2af4bc(0x4bd,0x445,0x3c7,0x1a6)+_0x2af4bc(0x564,0x1a7,0x408,0x1c9)+_0x2af4bc(-0x385,-0x3e5,-0xdb,-0xdc)+_0x26de55(0x20,0x2e5,-0x1ae,0x35b)+_0x26de55(0x3b5,0x4b8,0x5ef,0x499)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,-0xab,0x91,0x7a)+_0x26de55(-0xad,-0x404,-0x1e0,-0x141)+_0x2af4bc(-0x26d,-0x171,-0x39,-0x106)+'\x20(ta\x20!==\x20s'+'elf)\x20{\x0a\x20\x20\x20'+_0x26de55(-0xad,0x56,-0x36b,-0x128)+_0x2af4bc(0x324,-0xff,0x24,-0x1a5)+_0x2af4bc(-0x29c,0x28f,0x24,0x147)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x3d3,0x6f3,0x27e,0x299)+'yle.height'+_0x2af4bc(0xcf,0x1e3,0x215,-0x2b)+'Height\x20+\x20\x27'+_0x26de55(0x349,0x121,0x497,0x106)+_0x26de55(-0xad,-0x360,-0x3c7,-0x83)+_0x26de55(-0xad,-0x121,-0x359,-0x17c)+_0x26de55(-0xad,-0x2af,0x199,-0x2d5)+_0x26de55(0x3d8,0x514,0x4ec,0x12f)+'\x0a\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0xa3,-0x5,0x24,-0x1c5)+_0x26de55(-0xad,-0x22b,-0x394,0x28b)+_0x26de55(0x46b,0x538,0x5db,0x3f6)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x2f0,0x17d,0x24,0x213)+_0x26de55(-0xad,-0x2f0,-0x305,0x6f)+_0x26de55(-0x2b,-0x295,-0xa7,0x271)+_0x26de55(-0xad,-0xd,0x4,-0x18c)+_0x26de55(-0xad,0x98,-0x2b6,-0x245)+_0x26de55(-0x4a,0x21c,0x297,0x66)+_0x2af4bc(0x300,0x363,0x24,0x66)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(-0x4a,0xc8,0xe1,-0x141)+_0x2af4bc(0x383,0x7a4,0x48f,0x362)+_0x26de55(-0xad,-0x35f,0x1ad,-0x3a9)+'\x20\x20\x20\x20\x20\x20\x20}\x0a\x20'+_0x26de55(-0xad,-0xb3,-0x19,0xdf)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20}'+_0x2af4bc(0x216,0x3f7,0x48f,0x284)+_0x2af4bc(-0x13c,0x1c7,0x24,-0x36)+_0x26de55(0x2f4,-0x14,0x64b,0x396)+_0x26de55(-0xad,0xdf,0x1c5,0x84)+_0x26de55(0x513,0x529,0x240,0x221)+_0x26de55(0xde,0x1b4,0x1c9,0xab)+_0x2af4bc(0x65c,0x25b,0x506,0x772)+_0x2af4bc(0x198,0x16d,-0x21,0x42)+_0x2af4bc(0x205,0x29a,0x22f,0x242)+_0x26de55(0x3b5,0x5ae,0x3ee,0x153)+_0x2af4bc(-0x217,0x379,0x24,-0x2c)+_0x26de55(-0x10a,-0x384,0x1ef,-0x13e)+'\x20(this._ch'+_0x2af4bc(0x7e3,0x81c,0x541,0x5c4)+_0x2af4bc(0x231,0x41c,0x494,0x679)+_0x2af4bc(-0x1d2,0xe2,0x24,0xfa)+_0x26de55(-0xad,-0xc9,-0x1bd,-0x157)+_0x2af4bc(0x8ff,0x5c3,0x5d9,0x570)+'terval(thi'+_0x26de55(0x21d,0x342,0x233,0x32d)+_0x26de55(0x3da,0x122,0x308,0x19f)+_0x2af4bc(-0x241,0x311,0x24,0x28f)+_0x2af4bc(-0x34,0x300,0x24,0x94)+_0x2af4bc(0x2b5,0x2e5,0x10,0x36d)+'s._checkIn'+_0x26de55(0x146,0x439,-0xe2,0x47e)+'ull;\x0a\x20\x20\x20\x20\x20'+_0x2af4bc(-0xc1,-0x133,0x24,0x5a)+_0x26de55(0x3d8,0x53d,0x700,0x5a0)+'\x0a\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,0x2a4,-0x1ae,0x116)+_0x2af4bc(0x603,0x6de,0x53c,0x886)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0x2b,-0x1e2,0x208,-0x295)+_0x26de55(0x14c,-0x106,0x40b,0x1a1)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x608,0x33b,0x31e,0xd2)+'area.oninp'+_0x26de55(0x325,0x5f8,0x19a,0x576)+_0x2af4bc(-0x26f,-0x342,-0x8b,-0x7f)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x8d,-0x24e,0x170,-0x107)+_0x26de55(-0xbc,0x1bb,0x24f,0x55)+_0x2af4bc(0x30a,0x692,0x445,0x34a)+_0x26de55(0x4dc,0x200,0x5a9,0x804)+'s.dataset.'+'col,\x20this.'+_0x2af4bc(0x294,0x48b,0x44d,0x480)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x601,0x42a,0x52d,0x7c5)+_0x26de55(-0xad,0x22d,-0xae,-0x1b4)+_0x26de55(0x2f4,0x3b6,0x2f8,0x5f3)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x38e,0x54c,0x3f2,0x65a)+_0x2af4bc(0x3c2,0x1f8,0x4d9,0x368))+('rea);\x0a\x20\x20\x20\x20'+_0x26de55(-0xad,-0x1dd,-0x21f,0xde)+_0x2af4bc(0x167,0x3c9,0x46a,0x456)+'ndChild(td'+');\x0a\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x22b,0x7c,0x591,-0x3e)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x516,0x24e,0x1b5,0x1bc)+_0x2af4bc(-0x13,-0xdc,0x210,0x500)+_0x2af4bc(0x4a8,0x35d,0x277,0xb3)+_0x2af4bc(0x5fa,0x486,0x330,0x4d4)+_0x26de55(0x4da,0x6a3,0x31a,0x183)+_0x2af4bc(0x1aa,0x2bc,0x435,0x50b)+_0x26de55(0x136,0xb3,0x19f,0x3d5)+_0x2af4bc(-0x12a,0x212,0x231,-0x129)+'ttom\x20=\x20doc'+'ument.crea'+_0x26de55(0x342,0x3b6,0x85,0x61f)+_0x2af4bc(0x50f,0x3f2,0x2e3,0x630)+'\x20\x20\x20\x20\x20space'+_0x26de55(-0x71,-0x17,-0x1d4,-0x44)+'yle.height'+_0x26de55(0x10d,-0x64,0x236,-0xe6)+_0x26de55(0x479,0x2df,0x218,0x441)+_0x26de55(0x252,0x1d2,0x41a,0x420)+_0x26de55(0x401,0x53d,0x72f,0x62d)+_0x26de55(-0x145,0x1b,-0xb7,-0x466)+'\x27;\x0a\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(-0x322,-0x1a0,-0xc7,-0x4c)+_0x26de55(0x33f,0x3da,0x68c,0x2e0)+'d(spacerBo'+'ttom);\x0a\x20\x20\x20'+_0x2af4bc(0x30f,0x572,0x438,0x2c8)+_0x26de55(0x5f,-0x147,-0x2ea,-0x107)+_0x2af4bc(0x4d8,0x5d5,0x35d,0x439)+_0x26de55(0x3d1,0x15c,0x630,0x1d8)+_0x2af4bc(0x7b4,0x77b,0x559,0x6ab)+'\x20\x20\x20functio'+'n\x20updateCe'+_0x2af4bc(0x527,0x575,0x5e1,0x58c)+_0x26de55(-0x162,0x1fd,0x28,-0x19b)+_0x2af4bc(0x4b,0x29b,0x2d4,0xa2)+_0x26de55(0x3b3,0x672,0x1f4,0x643)+'.opener\x20&&'+_0x2af4bc(0x4b8,0x490,0x25c,0x450)+_0x26de55(0xa,-0x100,0x336,0x1d8)+_0x2af4bc(0xdf,0x1aa,0x197,0x1cb)+_0x2af4bc(0x249,0x197,0x2d4,0x3d8)+_0x26de55(0x255,0x561,0x2c7,0xad)+_0x2af4bc(0x4c2,0x1ab,0x425,0x5d9)+_0x2af4bc(-0x189,0x2c2,-0x3e,-0x300)+'l(parseInt'+_0x2af4bc(-0x23c,-0x36b,-0xa6,0xbf)+_0x26de55(0x4e5,0x50b,0x3cd,0x6ce)+_0x2af4bc(0x3a7,0x2a1,0x148,0x29f)+_0x2af4bc(0x2cb,0x349,-0x5,0x149)+'\x20\x20\x20\x20}\x0a\x0a\x20\x20\x20'+'\x20function\x20'+_0x2af4bc(0x2e4,0x2ea,0x46b,0x6c2)+_0x26de55(0x50,-0x1e2,0x14e,-0x184)+'window.ope'+_0x26de55(0x3a9,0x44b,0x373,0x34f)+_0x2af4bc(0x249,0x35e,0x138,0x2f3)+_0x26de55(0x2f7,0x5a1,0x2b3,0x634)+_0x2af4bc(-0xfe,0x3e3,0x8d,0x2a8)+'\x20\x20\x20\x20window'+_0x2af4bc(0x122,0x580,0x36b,0x631)+_0x2af4bc(0x30a,0x95,0x99,-0x289)+_0x26de55(0x3be,0x2e5,0x550,0x413)+_0x2af4bc(0x3c9,0x75e,0x578,0x534)+_0x26de55(-0x18,0x260,-0x220,0x32f)+_0x2af4bc(0x3ec,0x1e4,0x515,0x2b4)+'\x20and\x20Appli'+'ed!\x27);\x0a\x20\x20\x20'+_0x26de55(0x7e,0xaa,0xe2,0x2a)+'tion\x20extra'+_0x2af4bc(-0x4,-0x1d8,-0xc6,-0x2df)+_0x26de55(0x279,0x85,-0xac,0x56f)+_0x26de55(0x211,0x327,0x149,0x2d1)+_0x2af4bc(-0x7c,-0x1e8,-0xb8,0x20e)+_0x26de55(-0x1b0,-0x2a2,-0x168,-0x260)+'riggerExtr'+_0x26de55(-0x14,0x244,-0x211,-0x1b9)+_0x26de55(0x2ef,0xa7,0x4fd,0x315)+_0x26de55(0xb3,0x29b,0x26a,-0x156)+_0x2af4bc(0x1a2,0x469,0x3fd,0x52a)+'ggerExtrac'+_0x26de55(0x48c,0x2f8,0x424,0x2c9)+_0x2af4bc(0x64c,0x2c7,0x309,0x3b5)+_0x26de55(0x1bc,0xc1,0x32c,0x228)+_0x2af4bc(0x546,0x194,0x39b,0x16d)+_0x2af4bc(0x7c5,0x690,0x4d4,0x74a)+'f\x20(window.'+'opener\x20&&\x20'+'window.ope'+_0x2af4bc(0x4c9,0x4f5,0x51c,0x277)+_0x2af4bc(0x1d9,0x284,0xad,0x1bc)+_0x2af4bc(0x4a6,0x43b,0x38a,0x443)+_0x2af4bc(0x3a2,0x6a9,0x584,0x5b0)+_0x2af4bc(0x41e,0x599,0x339,0x277))+('\x20\x20\x20\x20\x20\x20\x20\x20wi'+'ndow.opene'+_0x2af4bc(0x7a,-0xf0,0x1cb,0x343)+_0x26de55(-0x178,-0x345,-0x32c,0x18f)+_0x2af4bc(-0x23e,0x276,-0xb5,-0x13f)+_0x26de55(0x2cc,0x179,0x7b,0x287)+'ateTableDa'+_0x26de55(0x36e,0x68c,0x507,0x43a)+'ion(data)\x20'+'{\x0a\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0xfb,0xc6,-0x1ee,0x253)+_0x2af4bc(0x5d8,0x6e3,0x5e9,0x543)+_0x2af4bc(0x74,0x217,0x2a9,0x2a8)+'\x20let\x20lastH'+_0x26de55(-0xc8,-0x49,-0x1a0,-0x40c)+_0x2af4bc(0x7af,0x3c9,0x4fa,0x781)+_0x26de55(0x460,0x49a,0x695,0x1da)+_0x2af4bc(0x1c0,-0x3b4,-0xb1,0x25c)+_0x2af4bc(0x555,0x7d6,0x4ff,0x55e)+_0x2af4bc(0x36b,0x333,0x1b8,0x362)+'earchInput'+_0x2af4bc(0x31c,0x3a5,0x46c,0x724)+'t.getEleme'+_0x26de55(0xd,0xc7,0x337,-0x335)+_0x26de55(0x3c6,0xbf,0x25c,0x201)+_0x26de55(-0x2f,0x1fe,-0xc9,0x205)+'\x20searchTer'+'m\x20=\x20search'+_0x2af4bc(-0x1d3,-0x18f,0x1e,-0x15d)+_0x2af4bc(0x2c2,0x31a,0x3c9,0x7e)+_0x26de55(-0x51,-0x36d,-0x3b3,-0x8c)+_0x2af4bc(0x22c,0x7a2,0x567,0x346)+_0x2af4bc(0x5a,0x290,0x156,0xc9)+'archTerm)\x20'+_0x2af4bc(-0x3b,0x46d,0x2d4,0x220)+_0x2af4bc(0x3d6,0x7e,0x131,0x0)+'ghlightedR'+_0x26de55(0xd0,0x13b,-0x5d,0x1ca)+_0x2af4bc(0x238,0x152,-0x6d,-0x1c1)+_0x2af4bc(0x1dc,-0x246,0x6d,0x23f)+_0x26de55(0x377,0xc7,0x2e3,0x6a4)+'ssList.rem'+'ove(\x27highl'+'ighted\x27);\x0a'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x421,0x1b1,0x305,0x24a)+_0x2af4bc(0xbb,0x4a3,0x325,0xd0)+'\x20=\x20null;\x0a\x20'+_0x26de55(0x131,0x18a,0x47f,0x499)+_0x2af4bc(0x2c9,0x1b0,0x169,0x1e1)+_0x2af4bc(0x2de,0x4a3,0x593,0x58b)+_0x26de55(0x2f9,0x513,0x36e,0x340)+_0x26de55(0x60,0xa6,0x2b2,0xba)+_0x26de55(0x2d1,0x384,0x556,0x16b)+_0x2af4bc(0xe4,0x2b8,0x1a1,0x2e6)+_0x2af4bc(0x207,0x1c2,0x1c5,0x2a9)+'ghlightedR'+_0x26de55(0x46,-0x24f,-0x4b,0x2e3)+_0x2af4bc(0xbe,0x9f,0x20d,0x15a)+_0x2af4bc(0x64a,0x4c2,0x3a9,0x64d)+'ed\x27);\x0a\x20\x20\x20\x20'+_0x26de55(0xf4,0x34d,-0x1a5,-0xa4)+'ghlightedR'+'ow\x20=\x20null;'+_0x26de55(0x4ea,0x601,0x67b,0x7b5)+_0x26de55(0xf9,0x10e,-0x141,0x18a)+'!currentDa'+_0x26de55(-0xb,0x29f,-0x1d5,0x163)+_0x2af4bc(0x3b4,0x5c1,0x5a5,0x325)+_0x26de55(0x195,0x27d,0xc4,0x36b)+_0x26de55(0x3b5,0x25e,0x1f7,0x5a9)+_0x26de55(0x316,0x3b0,0x3d4,0x5fa)+_0x26de55(0x133,-0x42,0x1ab,0x3ac)+_0x26de55(0x47a,0x49f,0x1d2,0x2aa)+_0x2af4bc(0x226,0x4c4,0x49f,0x42e)+_0x26de55(0x130,0x20a,-0x13e,0x16c)+_0x2af4bc(-0x36b,0x14b,-0x54,-0x5a)+_0x26de55(0x2a,-0x278,-0x2b4,0x35b)+_0x2af4bc(0x5fd,0x66,0x2eb,0x362)+_0x2af4bc(0x76,0x360,0x2ea,0x2a7)+'length;\x20i+'+'+)\x20{\x0a\x20\x20\x20\x20\x20'+_0x2af4bc(0x4cb,0x2f2,0x1a4,0x3fa)+_0x26de55(0x144,0x284,0x492,-0x1b5)+_0x2af4bc(-0x13c,0x90,0x1d0,0x407)+_0x26de55(0xed,0x2fd,-0x183,0x362)+_0x26de55(-0x139,-0x48f,0x123,-0x202)+'tinue;\x0a\x20\x20\x20'+_0x26de55(0x367,0x4b2,0x102,0x37c)+_0x26de55(-0x89,-0x8a,0xad,0x102)+_0x26de55(0x2f5,0x284,0x24,0x256)+_0x2af4bc(0x574,0x796,0x45a,0x535)+'ngth;\x20j++)'+_0x2af4bc(0x78f,0x2d7,0x490,0x75b)+_0x26de55(0x11d,-0x1f3,0x43a,0x409)+_0x26de55(0x2b,-0x20c,0x4b,-0x196)+_0x26de55(0xfe,0x87,-0x88,0xa3)+_0x2af4bc(0x300,0x4cc,0x22b,-0x6f)+_0x26de55(0x1d5,-0xe5,0x464,0x290)+_0x26de55(0x41a,0xe2,0x663,0x768))+(_0x2af4bc(0xc6,0x482,0x3c5,0x1d6)+_0x2af4bc(0x585,0x222,0x2d9,0x275)+_0x26de55(0x105,0x45d,0x22c,-0x1f5)+_0x26de55(0x62,0x152,0xf3,0x6)+_0x26de55(0x28f,0x4e8,0x2fa,0x66)+_0x2af4bc(0x232,0x30a,0x486,0x555)+_0x26de55(-0xad,0x274,-0xc6,0x17f)+'var\x20contai'+_0x26de55(0x2bf,0x20,0x12c,0x314)+_0x2af4bc(0x164,0x59,0x2a0,0x444)+'ementById('+_0x2af4bc(0x3c1,0x6b,0x22d,0x2c)+_0x26de55(0xf3,0x84,0xf8,-0x50)+_0x26de55(-0xad,-0x3cf,0x1ba,0x180)+_0x2af4bc(0x62f,0x259,0x2d0,0x293)+_0x2af4bc(-0x1f9,-0xc7,-0x87,-0x1e6)+_0x26de55(-0x16e,-0x2ef,-0x1c6,0xee)+'\x201)\x20*\x20rowH'+_0x2af4bc(0x135,0x1a8,0x2f4,0x362)+_0x26de55(-0xad,-0x80,0x3f,0x96)+_0x2af4bc(0x18e,0x492,0x42e,0x3e6)+'er.scrollT'+_0x26de55(0x4f3,0x4b5,0x5bc,0x5ff)+_0x26de55(0x206,0x238,0x3da,0x81)+'\x0a\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0x92,-0x2bd,0x211,-0x1ff)+_0x2af4bc(0x1a8,0x455,0x437,0x502)+_0x26de55(-0xeb,-0x3d0,-0xa4,-0x33e)+_0x26de55(0x41a,0xcd,0x753,0xe4)+_0x2af4bc(0x12d,-0x23e,0x24,0x21f)+_0x26de55(-0x164,-0x20d,-0x414,0x1d)+_0x2af4bc(0x45b,0x4d3,0x46c,0x5c4)+_0x2af4bc(0x1d1,0x48c,0x37e,0x223)+_0x26de55(0x1,-0x2eb,0x2f9,0x2)+'bleBody\x27);'+_0x26de55(0x41a,0x725,0x664,0x62e)+_0x2af4bc(-0xc6,0x10f,0x24,-0x2ea)+_0x26de55(-0x8a,0x173,-0x1be,0x278)+'=\x20tbody.qu'+_0x26de55(0x511,0x3f6,0x216,0x3b7)+_0x2af4bc(0x1e5,0xf7,0x22e,0x1ce)+_0x26de55(0x8c,0x2f,0x3d5,-0x2bd)+_0x26de55(-0x151,-0x9,0xca,-0x1d6)+_0x2af4bc(-0x1dc,-0x6e,0x24,-0x19)+_0x2af4bc(0x1b5,0x393,0xdc,0x3b3)+_0x26de55(-0xad,-0x376,0x2b8,0x150)+_0x26de55(0x311,0xf2,0x327,0x210)+_0x26de55(0x390,0x4d1,0x250,0x67e)+'h(function'+_0x26de55(0x457,0x114,0x502,0x1a4)+_0x26de55(0x41a,0x681,0x576,0x656)+_0x2af4bc(-0x322,0x6b,0x24,0x267)+_0x2af4bc(0x191,0x74,-0x88,0x8b)+_0x26de55(0x25,-0x2b2,0x15d,0x150)+_0x2af4bc(0xc4,0x27e,0x275,0x3f4)+'et.rowInde'+'x)\x20===\x20i)\x20'+_0x26de55(0x203,-0x87,0x553,-0x10c)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,-0x1bc,-0x5,-0x25b)+_0x2af4bc(0x208,0x218,-0x9a,-0x37a)+_0x26de55(0x453,0x4e3,0x5aa,0x210)+_0x2af4bc(0x561,0x718,0x532,0x333)+'ted\x27);\x0a\x20\x20\x20'+_0x2af4bc(-0x310,0xa6,0x24,0x13a)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x57,0x22,0x7b,0x19b)+_0x2af4bc(-0x2d7,0x2a4,0x9,-0x2a9)+'Row\x20=\x20domR'+_0x2af4bc(0xf0,0x2a7,0x3b0,0x538)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xd6,-0x22e,-0x2a2,0x177)+_0x2af4bc(-0x21f,0x2c2,0x24,0x185)+_0x26de55(-0xad,-0x40c,0x1f5,-0xdf)+'});\x0a\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x30,0x169,0x1dc,0x2db)+_0x26de55(-0xad,0xa9,0x257,0x24f)+'\x20\x20\x20\x20\x20\x0a\x20\x20\x20\x20'+_0x26de55(-0xad,0x1a,-0x20c,0xaa)+_0x26de55(0x423,0x51c,0x14f,0x615)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20}\x0a\x20\x20\x20\x20\x20\x20'+'\x20\x20}\x0a\x20\x20\x20\x20}\x0a'+'\x20\x20\x20\x20\x0a\x20\x20\x20\x20a'+'lert(\x27No\x20m'+_0x2af4bc(0x391,0x290,0x5eb,0x45d)+_0x26de55(0x46f,0x564,0x78d,0x1e2)+_0x26de55(0x190,-0x120,0xda,-0x111)+_0x26de55(0x102,-0x56,0x16e,-0xc)+_0x2af4bc(-0x2fe,-0x2f0,0x19,0x2ca)+_0x2af4bc(-0x24f,0x241,0x76,-0x1d7)+_0x2af4bc(-0x2b6,-0x291,-0x25,0x1a9)+'ener(\x27DOMC'+_0x26de55(-0xf9,-0x22b,-0x1b8,-0x1b0)+_0x2af4bc(0x2bf,0x477,0x2bb,0x290)+_0x2af4bc(0x26f,-0xee,-0x8b,-0x3c8)+_0x26de55(-0x8f,-0x1b0,-0x8a,-0x66)+_0x26de55(-0x165,-0x1e1,0x11a,0xa)+'put\x20=\x20docu')+(_0x2af4bc(0x40,0x45d,0x2a0,0x2d2)+_0x26de55(-0x13f,-0x46e,-0x1e1,-0x131)+_0x26de55(0x50c,0x31c,0x599,0x308)+'ut\x27);\x0a\x20\x20\x20\x20'+_0x26de55(0x6e,0x2d9,0x15b,-0x10f)+_0x26de55(0x1d0,-0x4c,-0xe,0x330)+_0x2af4bc(0x53d,0x5c5,0x490,0x141)+'\x20\x20\x20\x20\x20searc'+_0x2af4bc(0x162,0x2a7,0x1b0,0x38b)+_0x2af4bc(-0x1e6,-0x20,0xba,0x24e)+'ner(\x27keypr'+'ess\x27,\x20func'+_0x26de55(-0x7d,-0x220,-0xdf,-0x1f8)+_0x2af4bc(0x292,0x1dc,0x24,0x7b)+'\x20\x20\x20\x20\x20\x20if\x20('+_0x26de55(0x13b,-0xd2,-0x2c,-0x43)+_0x26de55(0x3a8,0x6f,0x258,0x317)+_0x26de55(0x41a,0x54c,0x17e,0xcd)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20searchTex'+_0x26de55(0x4f7,0x1f6,0x583,0x2a5)+_0x2af4bc(-0x88,0x30f,0x24,0x3a)+_0x2af4bc(-0x1c8,-0x114,0xa6,-0x171)+_0x2af4bc(0x9f,0x33,0x2fc,0x201)+_0x26de55(0x131,-0x211,0x314,0x2ee)+'\x20\x20\x20});\x0a\x0a\x20\x20'+_0x2af4bc(0x1be,0x2f3,0x75,0x304)+_0x26de55(0x23e,-0x112,0x553,0x91)+'ght\x20=\x20func'+_0x26de55(-0xcb,0x293,0x12c,-0x1a)+_0x26de55(-0x11e,-0x1f1,-0x2f1,-0x16e)+_0x26de55(0x3ec,0x6bc,0x346,0x520)+'a\x20=\x20null;\x0a'+_0x2af4bc(-0x279,0x265,0x56,-0x264)+_0x2af4bc(0x4f,0x281,0x6d,0x4e)+_0x2af4bc(-0x4b,0x111,0x43,-0x1dd)+'\x20\x20\x20\x20\x20\x20\x20\x20la'+_0x2af4bc(-0x2a7,0x3e,0x6d,0x18a)+_0x26de55(0x377,0x1fe,0xf1,0x262)+_0x26de55(0x19e,0x2b3,0x173,-0x16e)+_0x26de55(0x2a8,0x21e,0x1c0,0x358)+_0x2af4bc(0x12d,0x5c1,0x2a5,0x1c9)+_0x2af4bc(-0x29,-0x27e,-0x6d,0x130)+_0x2af4bc(-0x2ce,-0x282,0x6d,-0x2ef)+'tedRow\x20=\x20n'+'ull;\x0a\x20\x20\x20\x20}'+_0x2af4bc(0x1c5,0x262,-0x29,-0x2bb)+_0x2af4bc(0x1cc,0x2ab,0x9,0x6e)+_0x2af4bc(0x1d1,-0xd,0x11e,-0x172)+_0x26de55(0x300,0x8e,0x1eb,0x3ce)+_0x26de55(0xd4,-0x25f,0x303,-0x103)+_0x26de55(0x6d,-0x16f,-0x2ae,0x362)+_0x2af4bc(0x7f4,0x584,0x4ee,0x3f7)+_0x26de55(-0x56,0x8b,0x124,-0x1bb)+_0x26de55(0x92,-0x4c,-0x6a,0x215)+'||\x20typeof\x20'+_0x26de55(0x3c0,0x67d,0x382,0x424)+_0x26de55(-0x11a,-0x55,0x1d7,-0x366)+_0x2af4bc(0x231,0x506,0x558,0x57a)+_0x26de55(0x2bd,0x12e,0x21,0xb8)+_0x2af4bc(0x86f,0x457,0x58d,0x87b)+_0x2af4bc(0x552,0x282,0x3e8,0x1e0)+_0x2af4bc(0x656,0x4e8,0x3bf,0xd8)+_0x26de55(-0x22,-0x38,0x252,0x283)+');\x0a\x20\x20\x20\x20hig'+_0x26de55(0x3b8,0x690,0x628,0x315)+_0x2af4bc(0x67d,0x49a,0x492,0x216)+'xt.trim();'+_0x2af4bc(0xba,-0x199,0x153,0x382)+'\x20cleanText'+'\x20=\x20text.tr'+_0x2af4bc(0x726,0x56b,0x3cd,0x3ff)+'if\x20(curren'+'tData)\x20{\x0a\x20'+_0x26de55(0x2a0,0x5c8,0x55e,0x5d7)+_0x26de55(0x4b0,0x3b7,0x5b7,0x562)+'1;\x20i\x20<\x20cur'+_0x2af4bc(0x3b1,0x12b,0x230,0x1fa)+'ength;\x20i++'+')\x20{\x0a\x20\x20\x20\x20\x20\x20'+_0x26de55(-0x8f,0x1fa,-0x3cc,0x298)+_0x2af4bc(0x411,0x1d9,0x52a,0x7e6)+'rrentData['+_0x26de55(0x83,0x390,-0x118,-0x2c8)+'\x20\x20\x20\x20\x20\x20if\x20('+'!row)\x20cont'+'inue;\x0a\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x0a\x20'+_0x26de55(-0xad,0x283,-0x9b,0x256)+_0x26de55(0x38a,0x5d,0x392,0x408)+'j\x20=\x200;\x20j\x20<'+_0x2af4bc(0x46e,0x51e,0x21a,0x297)+_0x2af4bc(0x583,0x49c,0x3d4,0x67c)+_0x26de55(-0xad,0x25a,0x119,-0x2a8)+'\x20\x20\x20\x20\x20\x20cons'+'t\x20cellValu'+'e\x20=\x20(row[j'+']\x20||\x20\x27\x27).t'+_0x2af4bc(0x2ac,0x2d6,0x205,0x2d0)+_0x26de55(-0xad,-0x15a,-0x1c7,0x21c))+(_0x26de55(0x2f4,0x511,0x5c7,0x3a9)+_0x2af4bc(-0x5d,0x29a,0x24,0x169)+_0x26de55(0x2a6,0x281,0x3f9,0x53a)+'lue\x20===\x20cl'+_0x26de55(-0x33,-0x314,0x1f2,-0x36f)+'\x0a\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,-0x232,-0x1ad,0x1dc)+_0x2af4bc(0x435,0x24,0x11f,0x145)+_0x26de55(0x214,0x11c,0x200,0x3c9)+'querySelec'+_0x2af4bc(0x2de,0x211,0x29a,0x26e)+_0x26de55(0x5d,-0xf,-0xb9,-0x5)+_0x2af4bc(0x42e,0x53c,0x317,0x14f)+_0x2af4bc(-0x2f8,0x13e,0x24,-0x114)+_0x26de55(-0xad,-0x43,0x15a,-0x1fd)+_0x2af4bc(0x2a3,-0xd7,-0x60,-0x1e8)+'ow\x20=\x20null;'+_0x26de55(0x41a,0x2ed,0x335,0x127)+_0x26de55(-0xad,-0x3cb,-0x6f,-0x125)+_0x2af4bc(0x609,0x45f,0x5e7,0x3dd)+_0x26de55(-0xad,-0x1c8,-0x218,0x210)+_0x26de55(0x3db,0x6f4,0x214,0x650)+_0x2af4bc(0x25,0x243,0x31d,0x3b4)+_0x26de55(0x120,-0xe3,0x39d,0x451)+_0x2af4bc(0x7d8,0x602,0x486,0x602)+_0x26de55(-0xad,-0x396,0xad,-0x2f3)+_0x2af4bc(0x16c,-0x2c8,-0x39,-0xed)+_0x26de55(0x1ce,0xf7,0x3ad,0xdc)+_0x2af4bc(0x67,-0x4b,-0xd4,0x10b)+_0x2af4bc(0x382,0xb7,0x14c,0x40a)+_0x26de55(0x3c9,0x43f,0x2e5,0x46f)+_0x26de55(0x293,0x167,0x150,0x378)+_0x2af4bc(0xe3,-0x213,0x24,0x36)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x7d9,0x53d,0x569,0x839)+_0x26de55(0x3d2,0x3ca,0x65b,0x6c4)+';\x0a\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,-0xe8,0x48,-0x1cd)+_0x2af4bc(0x3bd,0x398,0x25f,0x283)+_0x2af4bc(-0x12e,0x2f5,0x24,-0x25d)+_0x26de55(-0x1e,-0x4,0x2d1,0x279)+_0x2af4bc(0x2e7,0x2b5,0x1d4,-0x13a)+_0x26de55(-0xad,-0x1bc,-0x3e6,-0x2f)+'\x20\x20\x0a\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x25,-0x298,0x24,0x2f8)+'\x20\x20\x20if\x20(fou'+'ndRow)\x20{\x0a\x20'+_0x2af4bc(-0x2ff,0x188,0x24,0xff)+_0x26de55(-0xad,-0x263,-0x269,0x171)+_0x2af4bc(0x503,0x2ad,0x328,0x636)+_0x2af4bc(0x458,0x507,0x5a7,0x809)+'edRow\x20&&\x20l'+_0x26de55(-0xb6,-0x27e,0x9b,-0x27c)+_0x2af4bc(0xde,0x2d,0x2df,0x7f)+_0x26de55(0x500,0x3f1,0x43a,0x37e)+_0x2af4bc(0x3dd,0x25c,0x486,0x7e2)+_0x26de55(-0xad,-0x1f8,-0x23f,0x238)+_0x2af4bc(-0x1a3,-0x134,0x24,0x26e)+_0x2af4bc(0x578,0x1e5,0x4f2,0x697)+_0x26de55(0x254,0x138,0x2a4,0x5b2)+_0x2af4bc(0x2ab,0x620,0x2bc,0x415)+_0x2af4bc(0x0,0x1f2,-0x34,0x7f)+'ighlighted'+'\x27);\x0a\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x276,0x3e,0x24,-0x1da)+'\x20\x20\x20\x20\x20\x20\x20\x20}\x0a'+_0x26de55(-0xad,0x132,0x1e3,0x20d)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(-0x7d,0x48d,0x189,0x1b1)+_0x26de55(0x46,-0x2a3,-0xea,0x25e)+_0x26de55(0xe,0x1f4,-0x165,-0x21e)+_0x2af4bc(0x621,0x30a,0x54d,0x711)+_0x2af4bc(0x1ba,0x745,0x48f,0x615)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x34f,0x304,0x49b,0x598)+'tHighlight'+'edRow\x20=\x20fo'+_0x26de55(0x30d,0x180,0x4d4,0x284)+_0x26de55(-0xad,-0x25d,-0x263,0xc2)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0x69,-0x28d,0xa4,0xc)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20con'+'st\x20contain'+_0x26de55(0x24,0x310,0x28c,0x12b)+'ent.getEle'+_0x26de55(0x2ee,0x144,0x53a,0x93)+_0x26de55(0x340,0x4f3,0x3e2,0x2b2)+_0x2af4bc(0x21f,-0x2bf,-0x47,-0x266)+_0x26de55(-0xad,-0x387,-0xce,-0x363)+_0x2af4bc(-0x187,0x2e5,0x24,-0xe6)+_0x2af4bc(0x269,0x2d9,0x57c,0x535)+_0x26de55(0x438,0x192,0x572,0x396)+_0x26de55(0x184,0x474,0x35f,-0x3b)+_0x2af4bc(0x11f,0x19a,0x71,0x1d2)+_0x26de55(-0xad,-0xbd,-0xf0,-0x332)+_0x2af4bc(-0xdd,-0xe0,0x24,-0x31f)+_0x26de55(0x4d5,0x82f,0x368,0x294)+_0x26de55(-0x173,-0x281,-0x428,0x142)+_0x2af4bc(0x614,0x384,0x599,0x412))+(_0x26de55(0x475,0x545,0x758,0x47f)+_0x2af4bc(-0x2c0,0x48,0x65,-0x1b)+_0x2af4bc(-0x251,-0x93,0x24,0x2f0)+_0x26de55(-0xad,-0x170,-0x147,0x265)+_0x2af4bc(0x4d9,0xf9,0x20b,0x468)+_0x26de55(-0x158,-0x121,-0x42a,-0xee)+_0x2af4bc(0x3ef,0x53b,0x2ec,0x233)+_0x26de55(0x515,0x76b,0x455,0x665)+_0x2af4bc(0x158,0x503,0x460,0x670)+_0x2af4bc(0x149,0x355,0x1bc,0x23)+_0x2af4bc(-0xbc,-0x4,0x24,-0x32a)+_0x26de55(-0xad,0x258,0xe1,-0x24d)+_0x26de55(-0x155,-0x8f,-0xb3,-0x53)+_0x26de55(0x2c1,0x1b0,0xb1,0xf1)+_0x26de55(0x4e0,0x3e2,0x21b,0x3c2)+'rollPositi'+'on,\x20behavi'+_0x26de55(0x152,-0x217,-0xa3,0xdf)+_0x2af4bc(0x485,0x6c2,0x595,0x648)+_0x26de55(-0xad,-0x2a2,-0x2c3,0x247)+_0x26de55(-0x15b,0x1ec,-0x45a,-0x367)+_0x26de55(-0xcf,0x9f,0x1a5,-0x1a4)+_0x26de55(-0xad,0x233,0x1d9,-0x291)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(-0x2c0,0x1f6,0x4d,0x334)+'ainer\x20=\x20do'+_0x2af4bc(0x2fb,0x439,0x485,0x1f4)+'ElementByI'+_0x2af4bc(0xed,0x14c,-0x72,-0x142)+_0x26de55(-0x154,-0x39b,-0x2e7,-0x1ff)+'\x0a\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x18d,-0x2ef,0x24,-0x30e)+'\x20\x20\x20\x20\x20const'+_0x26de55(0x4c1,0x4ca,0x7b1,0x6fd)+_0x2af4bc(0x5aa,0x65c,0x4d3,0x71c)+_0x26de55(0x36c,0x43c,0x620,0x415)+_0x2af4bc(0x1ed,0x257,0x4bb,0x4a4)+_0x26de55(-0xad,-0x413,-0x312,-0x3c0)+_0x2af4bc(0x96,0x275,0x24,0xdc)+_0x26de55(0x22f,0x22c,0x36,0x229)+'ontainerHe'+'ight\x20=\x20con'+'tainer.cli'+'entHeight;'+_0x26de55(0x41a,0x42e,0x72a,0x119)+_0x2af4bc(0x245,0x28e,0x24,0x238)+_0x2af4bc(0x4,0x192,0x2fb,0xe6)+_0x26de55(0x308,0x10,0x382,0x54a)+_0x2af4bc(-0x120,0x47a,0x21e,0x40b)+_0x2af4bc(-0x29e,-0x26a,-0x7c,0x27d)+'tion\x20-\x20(co'+_0x2af4bc(-0x324,-0x3b0,-0xcd,0x69)+_0x26de55(0x2e6,0x4e6,0x3bd,0x5db)+_0x2af4bc(0x1a9,-0x1ff,0x24,0x299)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(-0x155,-0x1f4,0x6c,-0x1f7)+_0x26de55(-0xad,0x18d,-0x233,0x75)+_0x26de55(0x3b0,0x2f8,0x167,0x31a)+_0x26de55(0x220,-0xed,-0x12f,0x213)+'crollTo({\x20'+_0x26de55(0x41a,0x151,0x44f,0x172)+_0x2af4bc(-0x28a,0xa7,0x24,0x379)+_0x2af4bc(0x1f9,0x534,0x471,0x7d5)+_0x2af4bc(0x38e,0x3ad,0x56c,0x698)+_0x26de55(0x27b,0x3aa,0x3bd,0x5c9)+_0x26de55(0x520,0x43f,0x2d7,0x201)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20behavior:'+'\x20\x27smooth\x27\x20'+_0x2af4bc(0x70c,0x2fd,0x4eb,0x747)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x22b,0x437,-0x125,0x4fa)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x2f4,-0x5d,0x33d,0x3ea)+_0x26de55(-0xad,0x1d3,-0x3c8,-0x1c5)+_0x26de55(0x271,0x29e,0x275,0x417)+_0x2af4bc(0x399,0x31c,0x2a8,0x2b3)+'unction()\x20'+_0x26de55(0x203,0x48d,0x51d,0x115)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(-0x9,0x64,0x24,0x1d1)+_0x26de55(-0x190,-0x1d2,-0x23a,-0x1b3)+'\x20=\x20tbody.q'+_0x2af4bc(0x739,0x735,0x3e1,0x590)+'orAll(\x27tr['+_0x2af4bc(0x54e,0x5df,0x5df,0x504)+_0x2af4bc(0x14,-0x12a,0xfa,0x262)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(-0xc4,0x140,0x24,-0x198)+_0x2af4bc(-0x271,-0x236,-0x1e,-0x1d2)+_0x26de55(-0xad,0x4a,0xdb,-0x8c)+_0x26de55(-0xad,-0x3ef,-0x29a,-0x414)+_0x2af4bc(0x5e6,0x2af,0x39c,0x5f9)+_0x26de55(-0x1ac,-0x48a,-0x436,0x105)+_0x2af4bc(0x233,0x2ca,0x4bc,0x277)+_0x2af4bc(0x358,0x10d,0x327,0x400)+_0x26de55(-0xad,0x299,-0x3e9,0x28a)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20')+('\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x3ff,0x648,0x2df,0x3e7)+'nt(domRow.'+_0x26de55(-0x8c,-0x3bc,-0x4b,-0x37)+_0x26de55(0x35e,0x118,0x641,0xdc)+'=\x20i)\x20{\x0a\x20\x20\x20'+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,-0x195,-0x14a,0x225)+_0x26de55(-0xad,-0x87,-0x2cb,0x16d)+_0x2af4bc(0x3f5,0x1ee,0x328,0x4ff)+_0x2af4bc(0x5ad,0x4ca,0x5a7,0x7cb)+_0x26de55(0x20a,0x42e,0xaf,0x238)+'astHighlig'+_0x26de55(0x20e,0x4c4,0x497,0x3a7)+_0x2af4bc(0x38f,0x6d9,0x4b1,0x6a8)+_0x26de55(0x203,0x2f,0x1c0,0x245)+_0x26de55(-0xad,0xdf,-0x32d,0x2f)+_0x2af4bc(0x30c,0x172,0x24,0x1d3)+_0x2af4bc(0x174,-0x13f,0x24,-0x17c)+_0x2af4bc(0x4cf,0x1f1,0x4f2,0x596)+'lightedRow'+_0x2af4bc(0x41a,0x3b7,0x2bc,0x197)+_0x26de55(-0x105,-0x1ed,-0x254,-0x45b)+'ighlighted'+_0x2af4bc(0x1c8,-0x4a,0x132,0x38)+_0x2af4bc(0x251,-0x209,0x24,-0x12a)+_0x26de55(-0xad,0x27d,-0x342,-0x1db)+_0x2af4bc(-0x1c7,0x24d,0x24,-0xa3)+_0x26de55(0x188,0x1d1,-0x85,0x353)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x1b5,0x91,0x24,-0x35)+_0x2af4bc(0x14d,0x557,0x333,0x248)+_0x2af4bc(0x30,0x3fc,0x22a,-0x33)+_0x26de55(-0xdd,-0x42e,0x6b,-0x1d6)+'highlighte'+_0x26de55(0x266,0x15e,0x593,0x16e)+_0x2af4bc(0x338,-0x14,0x24,-0xb4)+_0x26de55(-0xad,-0x15c,-0x99,0xdf)+_0x2af4bc(-0x2c2,0xff,0x24,0xc3)+_0x26de55(0x1fc,0x62,-0x15d,0x407)+_0x2af4bc(0x1c7,0x4bc,0x16b,0x4b2)+'=\x20domRow;\x0a'+_0x26de55(-0xad,-0x18f,0x7f,0x12)+_0x2af4bc(0x264,-0x2a0,0x24,-0x335)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x6b9,0x660,0x57a,0x412)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(-0xad,0x2d,0x22f,-0x37a)+_0x2af4bc(0x375,0x14f,0x4ae,0x578)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x2a6,-0x344,0x24,0x101)+_0x2af4bc(0x1c2,0x1b1,0x101,-0xc4)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x26de55(0x3d8,0x66d,0x331,0x263)+'\x0a\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(-0x318,-0xef,0x24,0x233)+_0x26de55(0x516,0x757,0x2e3,0x7a3)+_0x2af4bc(0x61,-0x18f,0x24,-0x1e4)+'\x20\x20return;\x0a'+_0x2af4bc(0x344,-0x315,0x24,0x88)+_0x2af4bc(0x54e,0x568,0x25f,0x30c)+'\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20'+_0x2af4bc(0x23a,0x234,0x259,0x590)+_0x2af4bc(-0x27b,-0x285,0x4,-0x2eb)+_0x2af4bc(0xbf,0x403,0x192,0x1d2)+_0x2af4bc(0x480,0x4aa,0x1b1,0x4ab)+_0x26de55(0x4f0,0x590,0x249,0x641)+_0x2af4bc(0x5e9,0x839,0x4e2,0x7cd)+_0x2af4bc(0x1ac,0xf8,-0xce,-0x427)+_0x26de55(-0xd,-0x374,0x7a,0x1b5)+_0x2af4bc(-0x1cb,0x23b,0x12a,0x433)+'lKey\x20&&\x20e.'+'key\x20===\x20\x27s'+_0x2af4bc(0x5f9,0x36,0x35c,0x406)+'\x20\x20\x20e.preve'+_0x2af4bc(0x63b,0x495,0x57f,0x422)+');\x0a\x20\x20\x20\x20\x20\x20\x20'+'\x20saveCSV()'+_0x2af4bc(-0x174,0x17,0x41,-0x2b)+'\x20\x20if\x20(e.ct'+_0x26de55(-0xe6,0x247,-0x1ee,-0x232)+_0x2af4bc(0x25e,0x673,0x58f,0x8b7)+_0x2af4bc(0x11f,0x36d,0xcd,-0x108)+'\x20\x20\x20\x20e.prev'+_0x2af4bc(0x5ea,0x332,0x406,0x4fe)+_0x2af4bc(0x1d2,0x1aa,0x3af,0x5a6)+_0x2af4bc(-0x1e4,0x240,-0x2a,0x2d)+_0x26de55(0x1ad,0x40c,-0x13a,0x196)+_0x26de55(0x24b,0x37e,-0xa9,0x4e)+_0x26de55(0x3aa,0x157,0x2c4,0x6c2)+_0x26de55(0xaa,0x2e8,0x90,0x17f)+_0x26de55(0x262,-0x40,-0x55,0x53a)+_0x26de55(0x3b4,0x39f,0x3cc,0x27f)+_0x26de55(-0xaa,-0x12b,-0x5f,-0x309)+'d(\x27searchI'+_0x2af4bc(0x1de,0xc4,0xef,-0x1a8)+_0x2af4bc(0x36c,0x41e,0xda,-0xdd)+_0x2af4bc(0x24d,0x851,0x575,0x3cb)+_0x2af4bc(0x2f8,0x1db,0x1e3,0xae)+_0x26de55(0xc5,-0x1f5,0x3d7,0xed))+(_0x26de55(0x1bd,-0x2,-0x45,0x103)+'\x20')),VisualEditorWindow[_0x26de55(0xbe,0x172,0x75,0xdd)][_0x26de55(0x378,0x38e,0x30f,0x285)]();const _0x4782b8=_0x58fcda[_0x26de55(-0x11c,0xcd,0x7c,-0x440)](setInterval,()=>{function _0x3b61d4(_0x10abef,_0x510267,_0x5b6d0b,_0x2c334a){return _0x2af4bc(_0x10abef-0x13f,_0x510267-0x66,_0x510267-0x5c7,_0x2c334a);}function _0x3dc535(_0x225011,_0x15f59c,_0x361c40,_0x7c61fb){return _0x2af4bc(_0x225011-0x1d5,_0x15f59c-0xf3,_0x361c40-0x49e,_0x15f59c);}VisualEditorWindow[_0x3b61d4(0x754,0x8c4,0x936,0x594)+'eData']&&_0x58fcda['lgDwM'](typeof VisualEditorWindow[_0x3dc535(0x8f2,0x8c8,0x79b,0x8c8)+_0x3dc535(0xa3f,0x934,0x953,0xa18)],_0x58fcda[_0x3dc535(0x770,0x15c,0x4a1,0x46e)])&&(_0x58fcda[_0x3b61d4(0x4d7,0x5a4,0x86f,0x6d5)](clearInterval,_0x4782b8),loadCSVData());},-0x3*0xa51+-0x1*0x1133+-0x39*-0xda);_0x58fcda[_0x26de55(0x495,0x6ea,0x250,0x1b5)](setTimeout,()=>{function _0x2c8c3b(_0x36fa04,_0x1c7391,_0x26554e,_0xada6a0){return _0x26de55(_0x36fa04-0x4f5,_0x1c7391-0x1a6,_0x26554e,_0xada6a0-0xf2);}function _0x2c2673(_0x5f3621,_0x4154bd,_0x532ee8,_0x22928c){return _0x26de55(_0x5f3621-0x11e,_0x4154bd-0x16f,_0x4154bd,_0x22928c-0x69);}_0x58fcda[_0x2c8c3b(0x401,0x252,0x504,0x736)](clearInterval,_0x4782b8),VisualEditorWindow&&!VisualEditorWindow[_0x2c2673(0x42c,0x521,0x623,0x787)]&&VisualEditorWindow[_0x2c2673(0x34a,0x60d,0x41c,0x66b)+_0x2c8c3b(0x8d9,0x5f9,0x5d3,0xb44)]&&_0x58fcda[_0x2c2673(0x357,0x44c,0x5bc,0x3a2)](loadCSVData);},-0xba*-0x26+0x17c0+-0x2b8c);const _0x5ae295=setInterval(()=>{function _0x2ac812(_0x3eac3a,_0x2a337a,_0x1e84a5,_0x384b4b){return _0x26de55(_0x384b4b-0x3ed,_0x2a337a-0x190,_0x1e84a5,_0x384b4b-0xbb);}function _0x15c497(_0xdab647,_0x427474,_0x39508b,_0x256cb7){return _0x26de55(_0x39508b-0x408,_0x427474-0x25,_0x256cb7,_0x256cb7-0x2a);}VisualEditorWindow&&VisualEditorWindow[_0x2ac812(0x895,0x844,0x4bb,0x6fb)]&&(_0x58fcda['NmPGv'](clearInterval,_0x5ae295),_0x58fcda[_0x15c497(-0x50,0x5e3,0x283,0x1ce)](clearInterval,_0x4782b8));},-0x66*-0x2d+0x2*0x42d+0x1854*-0x1);}}function loadCSVData(){const _0x29c84b={};_0x29c84b[_0x5482c1(0x2f9,0x3fa,0x13a,0x3f7)]=function(_0x579db5,_0x452814){return _0x579db5||_0x452814;};function _0x5482c1(_0x473c5a,_0x241d5f,_0x25607e,_0x4c5b5a){return _0x5dcb(_0x4c5b5a-0x174,_0x25607e);}_0x29c84b[_0x5482c1(0xa8d,0x686,0x7eb,0x893)]=_0x5482c1(0x74d,0xab0,0x465,0x7a2)+'ges.csv';function _0x1a7750(_0x27c5d7,_0x488dfb,_0x56cba7,_0xcbdeb1){return _0x5dcb(_0x488dfb- -0x3a6,_0x56cba7);}_0x29c84b['yQTZB']='utf8';const _0x34ca79=_0x29c84b;if(_0x34ca79[_0x5482c1(0x499,0x189,0x67f,0x3f7)](!fs,!path))return;let _0x5e04df=_0x34ca79['qDZVR'];!fs['existsSync'](_0x5e04df)&&(_0x5e04df=path[_0x5482c1(0x9a3,0xc2c,0x908,0x9e0)](_0x5482c1(0x99d,0x920,0x687,0x94a),_0x1a7750(0x5e8,0x288,0x69,0x43b)+_0x5482c1(0x3ff,0x445,0x6f1,0x619)));if(!fs[_0x5482c1(0x526,0x265,0x1ac,0x4dd)](_0x5e04df))return;const _0x4bb80d=fs[_0x5482c1(0x4b2,0x79f,0x56a,0x4cd)+'nc'](_0x5e04df,_0x34ca79[_0x5482c1(0x824,0x859,0x7a2,0x93e)]);csvData=parseCSV(_0x4bb80d),VisualEditorWindow&&!VisualEditorWindow[_0x5482c1(0x497,0x9ce,0x923,0x7e7)]&&VisualEditorWindow[_0x1a7750(0x26b,0x1eb,0x4fb,0x37f)+_0x1a7750(0x223,0x3a3,0x2fb,0x19d)](csvData);}window[_0x255014(0x398,0x3ac,0xe4,0x581)+'a']=loadCSVData,window[_0x255014(0x8d4,0x6f9,0x7c7,0x951)+_0x255014(0x80b,0x879,0x5dc,0x64a)]=function(_0x1e35de,_0x588b26,_0x3316ab){csvData[_0x1e35de]&&(csvData[_0x1e35de][_0x588b26]=_0x3316ab);},window['reloadTran'+'slationsAf'+'terSave']=function(){const _0x4bc844={'GxIVY':_0x9c8642(0x46b,0x37b,0x6d9,0x529)+_0x9c8642(0x46e,0x115,0x101,0x1d1),'klJhL':function(_0x5c8ee6){return _0x5c8ee6();},'yBPAW':function(_0x161487,_0xcd1abd){return _0x161487(_0xcd1abd);}};function _0x9c8642(_0x523d8d,_0x2bb3a4,_0x49ea4a,_0xcd86c1){return _0x159b99(_0x523d8d-0xca,_0x2bb3a4-0x2d,_0xcd86c1- -0x307,_0x49ea4a);}const _0x2677a1=_0x4bc844[_0x5d96a3(0x12a,0x45f,0x30a,0x142)]['split']('|');function _0x5d96a3(_0x1e6a1f,_0x27ce9f,_0xce334a,_0x1b8f80){return _0x159b99(_0x1e6a1f-0x30,_0x27ce9f-0x87,_0x1b8f80- -0x1a0,_0x1e6a1f);}let _0x29ae61=0xc51*0x1+-0x1*-0x119f+-0x1df0;while(!![]){switch(_0x2677a1[_0x29ae61++]){case'0':autoRefreshPictures&&_0x4bc844[_0x9c8642(0x32c,-0x159,0x267,-0x2d)](refreshAllPictures);continue;case'1':translationCache={};continue;case'2':translations={};continue;case'3':_0x4bc844['klJhL'](applyLanguageSettings);continue;case'4':_0x4bc844[_0x9c8642(-0x181,0x38d,-0x227,0x52)](loadTranslations,currentLanguage);continue;case'5':wordTranslations={};continue;case'6':SceneManager[_0x5d96a3(0x3d1,0x70d,0x4ee,0x659)]&&SceneManager[_0x5d96a3(0x677,0x7e5,0x756,0x659)][_0x9c8642(0x302,-0x2dc,-0x4f,0x29)+_0x5d96a3(0x82c,0x848,0x741,0x6a7)]&&SceneManager[_0x9c8642(0x301,0x82d,0x4cc,0x4f2)][_0x9c8642(0x4d,-0x193,-0xde,0x29)+'Windows']();continue;case'7':wordTranslationRegexes['clear']();continue;}break;}},window[_0x255014(0x3fa,0x16c,0x4c2,0x316)+'e']=function(){const _0xcc52a5={};_0xcc52a5[_0x4b29d6(0x968,0xaa4,0xbb8,0xb61)]=function(_0x3a82a1,_0x24ae45){return _0x3a82a1+_0x24ae45;},_0xcc52a5[_0x4637b7(0x250,0x19d,-0xb5,0xa0)]='game_messa'+_0x4b29d6(0x51e,0x767,0x7ce,0x7a5),_0xcc52a5[_0x4b29d6(0xa92,0x8e9,0x88a,0xbd9)]='utf8';const _0x4cf2bc=_0xcc52a5;if(!fs||!path)return;function _0x4b29d6(_0x211e43,_0x2b619c,_0xf0bc59,_0x50cc14){return _0x159b99(_0x211e43-0xd6,_0x2b619c-0x3,_0x2b619c-0x1e5,_0x211e43);}function _0x4637b7(_0xfb1662,_0x34ce6b,_0x55a808,_0x566d04){return _0x159b99(_0xfb1662-0x13a,_0x34ce6b-0xb4,_0xfb1662- -0x49,_0x34ce6b);}let _0xd286a0=_0x4cf2bc[_0x4b29d6(0x27c,0x47e,0x68a,0x3c4)];!fs[_0x4b29d6(0x60e,0x62b,0x592,0x469)](_0xd286a0)&&(_0xd286a0=path[_0x4b29d6(0x9d7,0xb2e,0xd48,0x842)](_0x4637b7(0x86a,0x7db,0x967,0x8f7),_0x4cf2bc['ptXPK']));const _0x58363b='\ufeff';let _0xe1d0af=_0x58363b;csvData[_0x4637b7(0x309,-0x2c,0x584,0x5bd)]((_0x3aba4e,_0x23b6f2)=>{function _0x2ffbaa(_0x1f5c75,_0x3a526d,_0x5f5302,_0x184ec0){return _0x4637b7(_0x5f5302- -0x19e,_0x3a526d,_0x5f5302-0x5f,_0x184ec0-0xdb);}const _0xbd4eec=_0x3aba4e[_0x372675(0x207,0x141,0xe7,-0xa5)](_0x12e340=>escapeCSV(_0x12e340||''))[_0x2ffbaa(0x553,0x8ad,0x762,0x83a)](CSV_SEPARATOR);function _0x372675(_0x614fb5,_0x3909dd,_0x2c1f2f,_0x5f369b){return _0x4637b7(_0x3909dd- -0x2bf,_0x614fb5,_0x2c1f2f-0x1e6,_0x5f369b-0x195);}_0xe1d0af+=_0x4cf2bc['KijkJ'](_0xbd4eec,'\x0a');}),fs['writeFileS'+'ync'](_0xd286a0,_0xe1d0af,_0x4cf2bc['pzDXt']),window[_0x4b29d6(0x21a,0x50c,0x4ed,0x5c8)+_0x4b29d6(0x854,0x909,0x819,0x5d0)+_0x4b29d6(0xc0e,0x8db,0xbfd,0xa6d)]&&window[_0x4b29d6(0x7ce,0x50c,0x3c1,0x730)+_0x4637b7(0x6db,0x443,0x407,0x491)+'terSave']();},window[_0x159b99(0x918,0x791,0x7a2,0x540)+_0x255014(0x44b,0x620,0x1bf,0x6d4)+'es']=function(){function _0x3c42f2(_0x1456c5,_0x242470,_0x299fa8,_0x5e307c){return _0x255014(_0x242470- -0x4cf,_0x242470-0x18a,_0x299fa8,_0x5e307c-0x178);}const _0x3c8c6b={'xaYUo':function(_0x5cd140){return _0x5cd140();},'LoGKk':function(_0x3d456f){return _0x3d456f();},'beVsW':function(_0x3f8cb7,_0x4899e4,_0xe26f71){return _0x3f8cb7(_0x4899e4,_0xe26f71);}};_0x3c8c6b[_0x339d55(0x793,0x721,0x843,0x60c)](extractAllMessages);function _0x339d55(_0x5d311a,_0x5c3def,_0x40f14d,_0x523f45){return _0x159b99(_0x5d311a-0x1df,_0x5c3def-0xac,_0x5c3def-0x69,_0x5d311a);}_0x3c8c6b[_0x339d55(0xbfe,0x953,0xa95,0x9a1)](setTimeout,()=>{VisualEditorWindow&&!VisualEditorWindow['closed']&&_0x3c8c6b['xaYUo'](loadCSVData);},-0x1*0x19c1+-0x1*0x178e+0x372b*0x1);};let lastHighlightedText='',isMessageActive=![];function _0x159b99(_0x1354fa,_0x433b18,_0xd624c3,_0x5cb578){return _0x5dcb(_0xd624c3-0xdd,_0x5cb578);}const _Game_Message_processMessageBuffer_Visual=Game_Message[_0x255014(0x9bd,0xc34,0xca2,0xbcb)][_0x159b99(0x6e3,0xad8,0x7ef,0x555)+_0x159b99(0x7fc,0x286,0x567,0x374)];Game_Message['prototype'][_0x255014(0x8db,0x7bc,0x62b,0x91a)+'sageBuffer']=function(){function _0x14b110(_0x3b4297,_0x2dd252,_0x15a5e4,_0x3298cf){return _0x159b99(_0x3b4297-0x38,_0x2dd252-0x9d,_0x3b4297- -0xa1,_0x15a5e4);}function _0x1be1f7(_0x4784e0,_0x2b57bb,_0x13a0ac,_0x229f27){return _0x159b99(_0x4784e0-0x1b4,_0x2b57bb-0x18,_0x13a0ac- -0x3f5,_0x4784e0);}const _0xee6df6={};_0xee6df6['cieSA']=function(_0xb43ea5,_0x17d283){return _0xb43ea5!==_0x17d283;};const _0x59ff4a=_0xee6df6;if(this['messageBuf'+'fer']&&this[_0x14b110(0x3c7,0x133,0x24f,0x638)+_0x1be1f7(-0x68,0x1f2,0xf,0xe)]['length']>-0x137f+-0xb*-0xe8+0x987){const _0x959002=this[_0x14b110(0x3c7,0x163,0x257,0x8d)+_0x1be1f7(0x16c,0x292,0xf,0x253)][_0x14b110(0x8a8,0x54e,0x672,0x565)]('\x0a');_0x59ff4a[_0x1be1f7(-0xc,0x1bb,0x2eb,0x4b)](_0x959002,lastHighlightedText)&&(lastHighlightedText=_0x959002,isMessageActive=!![],VisualEditorWindow&&!VisualEditorWindow[_0x1be1f7(0x490,0x2da,0x35b,0x624)]&&VisualEditorWindow['highlightR'+'ow']&&VisualEditorWindow['highlightR'+'ow'](_0x959002));}return _Game_Message_processMessageBuffer_Visual[_0x14b110(0x415,0xd9,0x1c5,0x4da)](this);};const _Window_Message_terminateMessage=Window_Message[_0x159b99(0xbe4,0x9d5,0x8d1,0xac1)]['terminateM'+'essage'];Window_Message[_0x159b99(0x9e7,0x7f8,0x8d1,0xa53)]['terminateM'+_0x255014(0x8c0,0xb81,0xafc,0xacb)]=function(){function _0x143ce6(_0xd5c986,_0x56caa1,_0x5ae247,_0x382468){return _0x255014(_0x5ae247- -0x56e,_0x56caa1-0x1f3,_0x56caa1,_0x382468-0x1bf);}_Window_Message_terminateMessage[_0x143ce6(-0x2c7,0xdf,0x34,-0x75)](this);function _0x5cbebc(_0x283353,_0x29673c,_0x313ff9,_0x42b46c){return _0x159b99(_0x283353-0x192,_0x29673c-0x66,_0x29673c- -0x35a,_0x42b46c);}!$gameMessage[_0x5cbebc(0x4c1,0x1f7,0x373,0x326)]()&&(isMessageActive=![],lastHighlightedText='',VisualEditorWindow&&!VisualEditorWindow[_0x143ce6(0x5bf,0x5b,0x2ce,0x5df)]&&VisualEditorWindow[_0x143ce6(-0x270,0x1d6,-0x75,-0x68)+_0x143ce6(0x9e,0xe9,0x1d6,0x2c2)]&&VisualEditorWindow[_0x5cbebc(0x227,0xb3,0x94,0x14c)+_0x143ce6(-0xc5,0x52a,0x1d6,0x215)]());};

    const H_Scene_Map_update_Localization = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function () {
        H_Scene_Map_update_Localization.call(this);

        if (sheetEditorButton !== 'none' && Input.isTriggered(sheetEditorButton)) {
            createLocalizationVisualEditor();
        }
    };
})();